<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;

class AdminAuthController extends Controller
{
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Show the admin login form.
     */
    public function showLoginForm()
    {
        return view('admin.auth.login');
    }

    /**
     * Handle admin login request.
     */
    public function login(Request $request)
    {
        // Check for CSRF token issues
        if (!$request->hasValidSignature() && !hash_equals($request->session()->token(), $request->input('_token'))) {
            return back()->withErrors([
                'token' => 'Your session has expired. Please refresh the page and try again.',
            ])->withInput($request->except('password'));
        }

        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string|min:6',
        ]);

        $credentials = $request->only('email', 'password');
        $remember = $request->has('remember');

        // First, check if user exists
        $user = User::where('email', $credentials['email'])->first();

        if (!$user) {
            return back()->withErrors([
                'email' => 'No account found with this email address.',
            ])->withInput($request->except('password'));
        }

        // Check if user is admin
        if ($user->role !== 'admin') {
            return back()->withErrors([
                'email' => 'Access denied. This portal is for administrators only.',
            ])->withInput($request->except('password'));
        }

        // Check if admin account is active
        if (!$user->is_active) {
            return back()->withErrors([
                'email' => 'Your admin account has been deactivated. Please contact system administrator.',
            ])->withInput($request->except('password'));
        }

        // Attempt to authenticate
        if (Auth::attempt($credentials, $remember)) {
            $request->session()->regenerate();

            // Log admin login activity
            \Log::info('Admin login successful', [
                'admin_id' => Auth::id(),
                'admin_email' => Auth::user()->email,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'timestamp' => now()
            ]);

            return redirect()->intended(route('admin.dashboard'))->with('success', 'Welcome to Admin Portal!');
        }

        // Authentication failed
        return back()->withErrors([
            'password' => 'Invalid password. Please try again.',
        ])->withInput($request->except('password'));
    }

    /**
     * Handle admin logout request.
     */
    public function logout(Request $request)
    {
        // Log admin logout activity
        if (Auth::check()) {
            \Log::info('Admin logout', [
                'admin_id' => Auth::id(),
                'admin_email' => Auth::user()->email,
                'ip_address' => $request->ip(),
                'timestamp' => now()
            ]);
        }

        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')->with('success', 'You have been logged out successfully.');
    }

    /**
     * Show admin password reset form.
     */
    public function showForgotPasswordForm()
    {
        return view('admin.auth.forgot-password');
    }

    /**
     * Handle admin password reset request.
     */
    public function sendResetLink(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
        ]);

        $user = User::where('email', $request->email)->first();

        // Check if user is admin
        if ($user->role !== 'admin') {
            return back()->withErrors([
                'email' => 'Password reset is only available for admin accounts.',
            ]);
        }

        // Here you would typically send a password reset email
        // For now, we'll just return a success message
        return back()->with('success', 'Password reset instructions have been sent to your email address.');
    }

    /**
     * Verify admin credentials for sensitive operations.
     */
    public function verifyPassword(Request $request)
    {
        $request->validate([
            'password' => 'required|string',
        ]);

        if (!Auth::check() || Auth::user()->role !== 'admin') {
            return response()->json(['success' => false, 'message' => 'Unauthorized access.'], 403);
        }

        if (Hash::check($request->password, Auth::user()->password)) {
            return response()->json(['success' => true, 'message' => 'Password verified successfully.']);
        }

        return response()->json(['success' => false, 'message' => 'Invalid password.'], 422);
    }

    /**
     * Get admin session information.
     */
    public function getSessionInfo()
    {
        if (!Auth::check() || Auth::user()->role !== 'admin') {
            return response()->json(['authenticated' => false]);
        }

        return response()->json([
            'authenticated' => true,
            'admin' => [
                'id' => Auth::id(),
                'name' => Auth::user()->name,
                'email' => Auth::user()->email,
                'role' => Auth::user()->role,
                'last_login' => Auth::user()->updated_at,
            ],
            'session' => [
                'expires_at' => now()->addMinutes(config('session.lifetime')),
                'csrf_token' => csrf_token(),
            ]
        ]);
    }
}