<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\User;
use App\Models\Agent;
use App\Models\Office;
use App\Models\District;
use App\Models\DistrictPricing;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    public function dashboard()
    {
        $stats = [
            'total_orders' => Order::count(),
            'pending_orders' => Order::where('status', 'created')->count(),
            'in_transit_orders' => Order::where('status', 'in_transit')->count(),
            'delivered_orders' => Order::where('status', 'delivered')->count(),
            'total_users' => User::where('role', 'customer')->count(),
            'total_agents' => Agent::count(),
            'total_offices' => Office::count(),
            'total_revenue' => Order::where('payment_status', 'paid')->sum('final_price'),
        ];

        $recent_orders = Order::with(['user', 'pickupDistrict', 'dropDistrict'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        $monthly_revenue = Order::where('payment_status', 'paid')
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->selectRaw('MONTH(created_at) as month, YEAR(created_at) as year, SUM(final_price) as revenue')
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->limit(12)
            ->get();

        return view('admin.dashboard', compact('stats', 'recent_orders', 'monthly_revenue'));
    }

    public function orders(Request $request)
    {
        $query = Order::with(['user', 'pickupDistrict', 'dropDistrict', 'assignedAgent', 'assignedOffice']);

        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }

        if ($request->has('payment_status') && $request->payment_status != '') {
            $query->where('payment_status', $request->payment_status);
        }

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('tracking_code', 'like', "%{$search}%")
                  ->orWhere('sender_name', 'like', "%{$search}%")
                  ->orWhere('receiver_name', 'like', "%{$search}%")
                  ->orWhere('sender_phone', 'like', "%{$search}%")
                  ->orWhere('receiver_phone', 'like', "%{$search}%");
            });
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.orders.index', compact('orders'));
    }

    public function showOrder(Order $order)
    {
        $order->load(['user', 'pickupDistrict', 'dropDistrict', 'assignedAgent', 'assignedOffice']);
        return view('admin.orders.show', compact('order'));
    }

    public function updateOrderStatus(Request $request, Order $order)
    {
        $request->validate([
            'status' => 'required|in:created,picked,in_transit,delivered,cancelled',
            'notes' => 'nullable|string|max:500'
        ]);

        $order->status = $request->status;
        if ($request->notes) {
            $order->notes = $request->notes;
        }

        // Update timestamps based on status
        if ($request->status == 'picked' && !$order->picked_at) {
            $order->picked_at = now();
        } elseif ($request->status == 'delivered' && !$order->delivered_at) {
            $order->delivered_at = now();
        }

        $order->save();

        return redirect()->back()->with('success', 'Order status updated successfully!');
    }

    public function assignAgent(Request $request, Order $order)
    {
        $request->validate([
            'agent_id' => 'required|exists:agents,id'
        ]);

        $order->assigned_agent_id = $request->agent_id;
        $order->save();

        return redirect()->back()->with('success', 'Agent assigned successfully!');
    }

    public function agents(Request $request)
    {
        $query = Agent::with(['user', 'assignedOrders']);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            })->orWhere('agent_code', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%");
        }

        $agents = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.agents.index', compact('agents'));
    }

    public function createAgent()
    {
        $users = User::where('role', 'customer')->whereDoesntHave('agent')->get();
        $offices = Office::where('is_active', true)->orderBy('name')->get();
        return view('admin.agents.create', compact('users', 'offices'));
    }

    public function storeAgent(Request $request)
    {
        \Log::info('Agent creation started', ['request_data' => $request->all()]);
        
        try {
            \Log::info('About to start agent validation');
            
            $validator = \Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'office_id' => 'required|exists:offices,id',
                'agent_code' => 'required|unique:agents,agent_code',
                'phone' => 'required|string|max:20',
                'address' => 'required|string|max:500',
                'vehicle_type' => 'required|in:bike,car,van,truck',
                'license_number' => 'required|string|max:50',
                'is_active' => 'nullable'
            ]);

            \Log::info('Agent validator created, checking for failures');

            if ($validator->fails()) {
                \Log::error('Agent validation failed', ['errors' => $validator->errors()->toArray()]);
                return redirect()->back()->withErrors($validator)->withInput();
            }

            \Log::info('Agent validation passed');

            \Log::info('About to create agent with data', [
                'user_id' => $request->user_id,
                'office_id' => $request->office_id,
                'agent_code' => $request->agent_code,
                'phone' => $request->phone,
                'address' => $request->address,
                'vehicle_type' => $request->vehicle_type,
                'license_number' => $request->license_number,
                'is_active' => $request->has('is_active') ? 1 : 0
            ]);

            $agent = Agent::create([
                'user_id' => $request->user_id,
                'office_id' => $request->office_id,
                'agent_code' => $request->agent_code,
                'phone' => $request->phone,
                'address' => $request->address,
                'vehicle_type' => $request->vehicle_type,
                'license_number' => $request->license_number,
                'is_active' => $request->has('is_active') ? 1 : 0
            ]);

            \Log::info('Agent created successfully', ['agent_id' => $agent->id]);

            // Update user role to agent
            \Log::info('About to update user role', ['user_id' => $request->user_id]);
            User::find($request->user_id)->update(['role' => 'agent']);
            \Log::info('User role updated to agent');

            return redirect()->route('admin.agents')->with('success', 'Agent created successfully!');
        } catch (\Exception $e) {
            \Log::error('Agent creation failed', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return redirect()->back()->withInput()->with('error', 'Failed to create agent: ' . $e->getMessage());
        }
    }

    public function editAgent(Agent $agent)
    {
        return view('admin.agents.edit', compact('agent'));
    }

    public function updateAgent(Request $request, Agent $agent)
    {
        $request->validate([
            'phone' => 'required|string|max:20',
            'address' => 'required|string|max:500',
            'vehicle_type' => 'required|in:bike,car,van,truck',
            'license_number' => 'required|string|max:50',
            'is_active' => 'boolean'
        ]);

        $agent->update([
            'phone' => $request->phone,
            'address' => $request->address,
            'vehicle_type' => $request->vehicle_type,
            'license_number' => $request->license_number,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.agents')->with('success', 'Agent updated successfully!');
    }

    public function offices(Request $request)
    {
        $query = Office::with(['district']);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('address', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
        }

        $offices = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.offices.index', compact('offices'));
    }

    public function createOffice()
    {
        $districts = District::orderBy('name')->get();
        return view('admin.offices.create', compact('districts'));
    }

    public function storeOffice(Request $request)
    {
        \Log::info('Office creation started', ['request_data' => $request->all()]);
        
        // Check if validation is even being reached
        \Log::info('About to start validation');
        
        try {
            $validator = \Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'district_id' => 'required|exists:districts,id',
                'address' => 'required|string|max:500',
                'phone' => 'required|string|max:20',
                'email' => 'nullable|email|max:255',
                'manager_name' => 'required|string|max:255',
                'is_active' => 'nullable'
            ]);

            \Log::info('Validator created, checking for failures');

            if ($validator->fails()) {
                \Log::error('Office validation failed', ['errors' => $validator->errors()->toArray()]);
                return redirect()->back()->withErrors($validator)->withInput();
            }

            \Log::info('Office validation passed');

            \Log::info('About to create office with data', [
                'name' => $request->name,
                'district_id' => $request->district_id,
                'address' => $request->address,
                'phone' => $request->phone,
                'email' => $request->email,
                'manager_name' => $request->manager_name,
                'is_active' => $request->has('is_active') ? 1 : 0
            ]);

            $office = Office::create([
                'name' => $request->name,
                'district_id' => $request->district_id,
                'address' => $request->address,
                'phone' => $request->phone,
                'email' => $request->email,
                'manager_name' => $request->manager_name,
                'is_active' => $request->has('is_active') ? 1 : 0
            ]);

            \Log::info('Office created successfully', ['office_id' => $office->id]);

            return redirect()->route('admin.offices')->with('success', 'Office created successfully!');
        } catch (\Exception $e) {
            \Log::error('Office creation failed', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return redirect()->back()->withInput()->with('error', 'Failed to create office: ' . $e->getMessage());
        }
    }

    public function editOffice(Office $office)
    {
        $districts = District::orderBy('name')->get();
        return view('admin.offices.edit', compact('office', 'districts'));
    }

    public function updateOffice(Request $request, Office $office)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'district_id' => 'required|exists:districts,id',
            'address' => 'required|string|max:500',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'manager_name' => 'required|string|max:255',
            'is_active' => 'boolean'
        ]);

        $office->update([
            'name' => $request->name,
            'district_id' => $request->district_id,
            'address' => $request->address,
            'phone' => $request->phone,
            'email' => $request->email,
            'manager_name' => $request->manager_name,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.offices')->with('success', 'Office updated successfully!');
    }

    public function districts(Request $request)
    {
        $query = District::withCount(['pickupOrders', 'dropOrders', 'offices']);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('province', 'like', "%{$search}%");
        }

        $districts = $query->orderBy('name')->paginate(20);

        return view('admin.districts.index', compact('districts'));
    }

    public function createDistrict()
    {
        return view('admin.districts.create');
    }

    public function storeDistrict(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:districts,name',
            'province' => 'required|string|max:255',
            'is_active' => 'boolean'
        ]);

        District::create([
            'name' => $request->name,
            'province' => $request->province,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.districts')->with('success', 'District created successfully!');
    }

    public function editDistrict(District $district)
    {
        return view('admin.districts.edit', compact('district'));
    }

    public function updateDistrict(Request $request, District $district)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:districts,name,' . $district->id,
            'province' => 'required|string|max:255',
            'is_active' => 'boolean'
        ]);

        $district->update([
            'name' => $request->name,
            'province' => $request->province,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.districts')->with('success', 'District updated successfully!');
    }

    public function destroyDistrict(District $district)
    {
        // Check if district has associated orders or offices
        if ($district->pickupOrders()->count() > 0 || $district->dropOrders()->count() > 0 || $district->offices()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete district. It has associated orders or offices.');
        }

        $district->delete();
        return redirect()->route('admin.districts')->with('success', 'District deleted successfully!');
    }

    public function reports(Request $request)
    {
        $period = $request->get('period', 'month');
        $startDate = $request->get('start_date', Carbon::now()->subMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', Carbon::now()->format('Y-m-d'));

        // Revenue Report
        $revenue_data = Order::where('payment_status', 'paid')
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, SUM(final_price) as revenue, COUNT(*) as orders')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Order Status Report
        $status_data = Order::whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->get();

        // Top Districts Report
        $district_data = Order::with(['pickupDistrict', 'dropDistrict'])
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get()
            ->flatMap(function($order) {
                return [
                    ['district' => $order->pickupDistrict->name, 'type' => 'pickup'],
                    ['district' => $order->dropDistrict->name, 'type' => 'drop']
                ];
            })
            ->groupBy('district')
            ->map(function($group) {
                return $group->count();
            })
            ->sortDesc()
            ->take(10);

        // Agent Performance
        $agent_performance = Agent::with('user')
            ->withCount(['assignedOrders' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }])
            ->having('assigned_orders_count', '>', 0)
            ->orderBy('assigned_orders_count', 'desc')
            ->limit(10)
            ->get();

        return view('admin.reports', compact(
            'revenue_data', 'status_data', 'district_data', 'agent_performance',
            'startDate', 'endDate', 'period'
        ));
    }

    public function users(Request $request)
    {
        $query = User::withCount(['orders']);

        if ($request->has('search') && $request->search != '') {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
        }

        if ($request->has('role') && $request->role != '') {
            $query->where('role', $request->role);
        }

        if ($request->has('status') && $request->status != '') {
            $isActive = $request->status === 'active';
            $query->where('is_active', $isActive);
        }

        $users = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    public function editUser(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    public function updateUser(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'role' => 'required|in:customer,agent,admin',
            'is_active' => 'boolean'
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'role' => $request->role,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.users')->with('success', 'User updated successfully!');
    }

    public function toggleUserStatus(Request $request, User $user)
    {
        $user->update([
            'is_active' => $request->is_active
        ]);

        $status = $request->is_active ? 'activated' : 'deactivated';
        return redirect()->back()->with('success', "User {$status} successfully!");
    }

    public function destroyUser(User $user)
    {
        // Prevent deletion of admin users
        if ($user->role === 'admin') {
            return redirect()->back()->with('error', 'Cannot delete admin users.');
        }

        // Check if user has orders
        if ($user->orders()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete user with existing orders.');
        }

        // Check if user is an agent with assignments
        if ($user->agent && $user->agent->assignedOrders()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete agent with assigned orders.');
        }

        $user->delete();
        return redirect()->route('admin.users')->with('success', 'User deleted successfully!');
    }

    // District Pricing Management Methods

    public function districtPricing()
    {
        $pricings = DistrictPricing::with(['pickupDistrict', 'dropDistrict'])
            ->orderBy('pickup_district_id')
            ->orderBy('drop_district_id')
            ->orderBy('service_type')
            ->paginate(20);

        $districts = District::orderBy('name')->get();
        $serviceTypes = DistrictPricing::getServiceTypes();

        return view('admin.pricing.index', compact('pricings', 'districts', 'serviceTypes'));
    }

    public function createDistrictPricing()
    {
        $districts = District::orderBy('name')->get();
        $serviceTypes = DistrictPricing::getServiceTypes();

        return view('admin.pricing.create', compact('districts', 'serviceTypes'));
    }

    public function storeDistrictPricing(Request $request)
    {
        $request->validate([
            'pickup_district_id' => 'required|exists:districts,id',
            'drop_district_id' => 'required|exists:districts,id',
            'service_type' => 'required|in:standard,express,overnight',
            'base_rate' => 'nullable|numeric|min:0',
            'per_kg_rate' => 'required|numeric|min:0',
            'minimum_charge' => 'nullable|numeric|min:0',
            'maximum_charge' => 'nullable|numeric|min:0',
            'estimated_days' => 'required|integer|min:1',
            'is_active' => 'boolean',
            'notes' => 'nullable|string|max:1000',
        ]);

        // Check for existing pricing for same route and service
        $existing = DistrictPricing::where('pickup_district_id', $request->pickup_district_id)
            ->where('drop_district_id', $request->drop_district_id)
            ->where('service_type', $request->service_type)
            ->first();

        if ($existing) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Pricing already exists for this route and service type.');
        }

        DistrictPricing::create([
            'pickup_district_id' => $request->pickup_district_id,
            'drop_district_id' => $request->drop_district_id,
            'service_type' => $request->service_type,
            'base_rate' => $request->base_rate,
            'per_kg_rate' => $request->per_kg_rate,
            'minimum_charge' => $request->minimum_charge,
            'maximum_charge' => $request->maximum_charge,
            'estimated_days' => $request->estimated_days,
            'is_active' => $request->has('is_active'),
            'notes' => $request->notes,
        ]);

        return redirect()->route('admin.pricing')->with('success', 'District pricing created successfully!');
    }

    public function editDistrictPricing(DistrictPricing $pricing)
    {
        $districts = District::orderBy('name')->get();
        $serviceTypes = DistrictPricing::getServiceTypes();

        return view('admin.pricing.edit', compact('pricing', 'districts', 'serviceTypes'));
    }

    public function updateDistrictPricing(Request $request, DistrictPricing $pricing)
    {
        $request->validate([
            'pickup_district_id' => 'required|exists:districts,id',
            'drop_district_id' => 'required|exists:districts,id',
            'service_type' => 'required|in:standard,express,overnight',
            'base_rate' => 'nullable|numeric|min:0',
            'per_kg_rate' => 'required|numeric|min:0',
            'minimum_charge' => 'nullable|numeric|min:0',
            'maximum_charge' => 'nullable|numeric|min:0',
            'estimated_days' => 'required|integer|min:1',
            'is_active' => 'boolean',
            'notes' => 'nullable|string|max:1000',
        ]);

        // Check for existing pricing for same route and service (excluding current)
        $existing = DistrictPricing::where('pickup_district_id', $request->pickup_district_id)
            ->where('drop_district_id', $request->drop_district_id)
            ->where('service_type', $request->service_type)
            ->where('id', '!=', $pricing->id)
            ->first();

        if ($existing) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Pricing already exists for this route and service type.');
        }

        $pricing->update([
            'pickup_district_id' => $request->pickup_district_id,
            'drop_district_id' => $request->drop_district_id,
            'service_type' => $request->service_type,
            'base_rate' => $request->base_rate,
            'per_kg_rate' => $request->per_kg_rate,
            'minimum_charge' => $request->minimum_charge,
            'maximum_charge' => $request->maximum_charge,
            'estimated_days' => $request->estimated_days,
            'is_active' => $request->has('is_active'),
            'notes' => $request->notes,
        ]);

        return redirect()->route('admin.pricing')->with('success', 'District pricing updated successfully!');
    }

    public function destroyDistrictPricing(DistrictPricing $pricing)
    {
        $pricing->delete();
        return redirect()->route('admin.pricing')->with('success', 'District pricing deleted successfully!');
    }

    public function togglePricingStatus(Request $request, DistrictPricing $pricing)
    {
        $pricing->update([
            'is_active' => $request->is_active
        ]);

        $status = $request->is_active ? 'activated' : 'deactivated';
        return redirect()->back()->with('success', "Pricing {$status} successfully!");
    }

    // API method for getting pricing information
    public function getPricing(Request $request)
    {
        $request->validate([
            'pickup_district_id' => 'required|exists:districts,id',
            'drop_district_id' => 'required|exists:districts,id',
            'service_type' => 'nullable|in:standard,express,overnight',
            'weight' => 'nullable|numeric|min:0.1',
        ]);

        $serviceType = $request->service_type ?? 'standard';
        $weight = $request->weight ?? 1;

        $pricing = DistrictPricing::getPricing(
            $request->pickup_district_id,
            $request->drop_district_id,
            $serviceType
        );

        if (!$pricing) {
            return response()->json([
                'success' => false,
                'message' => 'No pricing found for this route and service type.'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'pricing' => [
                'base_rate' => $pricing->base_rate,
                'per_kg_rate' => $pricing->per_kg_rate,
                'minimum_charge' => $pricing->minimum_charge,
                'maximum_charge' => $pricing->maximum_charge,
                'estimated_days' => $pricing->estimated_days,
                'calculated_price' => $pricing->calculatePrice($weight),
                'service_type' => $pricing->service_type,
                'pickup_district' => $pricing->pickupDistrict->name,
                'drop_district' => $pricing->dropDistrict->name,
            ]
        ]);
    }
}
