<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    /**
     * Show the login form
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'password' => 'required|string',
        ]);

        $loginField = $request->login;
        $password = $request->password;

        // Determine if login field is email or phone
        $fieldType = filter_var($loginField, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';

        // Find user by email or phone
        $user = User::where($fieldType, $loginField)->first();

        if (!$user || !Hash::check($password, $user->password)) {
            throw ValidationException::withMessages([
                'login' => ['The provided credentials are incorrect.'],
            ]);
        }

        if (!$user->is_active) {
            throw ValidationException::withMessages([
                'login' => ['Your account has been deactivated. Please contact support.'],
            ]);
        }

        Auth::login($user, $request->boolean('remember'));

        $request->session()->regenerate();

        return redirect()->intended(route('home'))->with('success', 'Welcome back!');
    }

    /**
     * Handle logout request
     */
    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home')->with('success', 'You have been logged out successfully.');
    }

    /**
     * Show phone OTP form
     */
    public function showPhoneForm()
    {
        return view('auth.phone');
    }

    /**
     * Send OTP to phone (Firebase integration placeholder)
     */
    public function sendOTP(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|max:20',
        ]);

        // In a real implementation, integrate with Firebase Auth
        // For now, return success message
        return response()->json([
            'success' => true,
            'message' => 'OTP sent successfully to ' . $request->phone
        ]);
    }

    /**
     * Verify OTP (Firebase integration placeholder)
     */
    public function verifyOTP(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|max:20',
            'otp' => 'required|string|size:6',
        ]);

        // In a real implementation, verify OTP with Firebase
        // For demo purposes, accept any 6-digit OTP
        if (strlen($request->otp) === 6) {
            // Find or create user
            $user = User::firstOrCreate(
                ['phone' => $request->phone],
                [
                    'name' => 'User ' . substr($request->phone, -4),
                    'role' => 'customer',
                    'is_active' => true,
                    'phone_verified_at' => now(),
                ]
            );

            Auth::login($user);

            return redirect()->route('home')->with('success', 'Phone verification successful!');
        }

        return back()->withErrors(['otp' => 'Invalid OTP. Please try again.']);
    }

    /**
     * Redirect to Google OAuth (placeholder)
     */
    public function redirectToGoogle()
    {
        // In a real implementation, use Laravel Socialite
        return redirect()->route('login')->with('info', 'Google authentication will be implemented with Socialite package.');
    }

    /**
     * Handle Google OAuth callback (placeholder)
     */
    public function handleGoogleCallback()
    {
        // In a real implementation, handle Google OAuth callback
        return redirect()->route('login')->with('info', 'Google authentication callback.');
    }

    /**
     * Redirect to Facebook OAuth (placeholder)
     */
    public function redirectToFacebook()
    {
        // In a real implementation, use Laravel Socialite
        return redirect()->route('login')->with('info', 'Facebook authentication will be implemented with Socialite package.');
    }

    /**
     * Handle Facebook OAuth callback (placeholder)
     */
    public function handleFacebookCallback()
    {
        // In a real implementation, handle Facebook OAuth callback
        return redirect()->route('login')->with('info', 'Facebook authentication callback.');
    }

    /**
     * Show forgot password form
     */
    public function showForgotPasswordForm()
    {
        return view('auth.forgot-password');
    }

    /**
     * Send password reset link
     */
    public function sendResetLink(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email'
        ]);

        // In a real implementation, send password reset email
        // For now, just return success message
        return back()->with('success', 'Password reset link sent to your email address.');
    }
}
