<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\District;
use App\Models\Office;
use App\Models\Agent;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CourierController extends Controller
{
    /**
     * Show courier booking form
     */
    public function create()
    {
        $districts = District::orderBy('name')->get();
        return view('courier.create', compact('districts'));
    }

    /**
     * Store new courier booking
     */
    public function store(Request $request)
    {
        $request->validate([
            'pickup_district_id' => 'required|exists:districts,id',
            'drop_district_id' => 'required|exists:districts,id',
            'sender_name' => 'required|string|max:255',
            'sender_phone' => 'required|string|max:20',
            'sender_address' => 'required|string',
            'receiver_name' => 'required|string|max:255',
            'receiver_phone' => 'required|string|max:20',
            'receiver_address' => 'required|string',
            'weight' => 'required|numeric|min:0.1|max:1000',
            'size' => 'required|in:Small,Medium,Large',
            'category' => 'required|string|max:100',
            'description' => 'nullable|string|max:500',
            'delivery_type' => 'required|in:standard,express,same_day',
            'vehicle_type' => 'required|in:bike,car,van,truck',
            'payment_method' => 'required|in:cod,jazzcash,easypaisa',
        ]);

        DB::beginTransaction();
        try {
            // Calculate estimated price using district-based pricing
            $estimatedPrice = Order::calculatePrice(
                $request->weight,
                $request->delivery_type,
                $request->vehicle_type,
                $request->pickup_district_id,
                $request->drop_district_id
            );

            // Find nearest office for pickup district
            $assignedOffice = Office::where('district_id', $request->pickup_district_id)
                                   ->where('is_active', true)
                                   ->first();

            // Create order
            $order = Order::create([
                'tracking_code' => Order::generateTrackingCode(),
                'user_id' => Auth::id(),
                'pickup_district_id' => $request->pickup_district_id,
                'drop_district_id' => $request->drop_district_id,
                'assigned_office_id' => $assignedOffice ? $assignedOffice->id : null,
                'sender_name' => $request->sender_name,
                'sender_phone' => $request->sender_phone,
                'sender_address' => $request->sender_address,
                'receiver_name' => $request->receiver_name,
                'receiver_phone' => $request->receiver_phone,
                'receiver_address' => $request->receiver_address,
                'weight' => $request->weight,
                'size' => $request->size,
                'category' => $request->category,
                'description' => $request->description,
                'delivery_type' => $request->delivery_type,
                'vehicle_type' => $request->vehicle_type,
                'estimated_price' => $estimatedPrice,
                'final_price' => $estimatedPrice,
                'payment_method' => $request->payment_method,
                'status' => 'created',
            ]);

            DB::commit();

            // Redirect based on payment method
            if ($request->payment_method === 'cod') {
                return redirect()->route('courier.confirmation', $order)
                               ->with('success', 'Order placed successfully! You will pay on delivery.');
            } else {
                // Redirect to payment processing
                return redirect()->route('payment.' . $request->payment_method, $order);
            }

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()->with('error', 'Failed to create order. Please try again.');
        }
    }

    /**
     * Show order confirmation
     */
    public function confirmation(Order $order)
    {
        // Ensure user can only see their own orders
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $order->load(['pickupDistrict', 'dropDistrict', 'assignedOffice']);
        return view('courier.confirmation', compact('order'));
    }

    /**
     * Show user's orders
     */
    public function index()
    {
        $orders = Order::where('user_id', Auth::id())
                      ->with(['pickupDistrict', 'dropDistrict'])
                      ->orderBy('created_at', 'desc')
                      ->paginate(10);

        return view('orders.index', compact('orders'));
    }

    /**
     * Show specific order details
     */
    public function show($orderId)
    {
        $order = Order::with(['pickupDistrict', 'dropDistrict', 'assignedOffice', 'assignedAgent.user'])->findOrFail($orderId);
        
        // Ensure user can only see their own orders
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        return view('orders.show', compact('order'));
    }

    /**
     * Generate invoice PDF
     */
    public function invoice(Order $order)
    {
        // Ensure user can only see their own orders
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        $order->load(['pickupDistrict', 'dropDistrict', 'assignedOffice']);
        return view('orders.invoice', compact('order'));
    }

    /**
     * Process JazzCash payment
     */
    public function processJazzCash(Order $order)
    {
        // Ensure user can only pay for their own orders
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        // In a real implementation, integrate with JazzCash API
        // For now, simulate payment processing
        $order->update([
            'payment_status' => 'paid',
            'payment_reference' => 'JC' . time() . rand(1000, 9999)
        ]);

        return redirect()->route('payment.success', $order);
    }

    /**
     * Process Easypaisa payment
     */
    public function processEasypaisa(Order $order)
    {
        // Ensure user can only pay for their own orders
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        // In a real implementation, integrate with Easypaisa API
        // For now, simulate payment processing
        $order->update([
            'payment_status' => 'paid',
            'payment_reference' => 'EP' . time() . rand(1000, 9999)
        ]);

        return redirect()->route('payment.success', $order);
    }

    /**
     * Payment success page
     */
    public function paymentSuccess(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        return view('payment.success', compact('order'));
    }

    /**
     * Payment failed page
     */
    public function paymentFailed(Order $order)
    {
        if ($order->user_id !== Auth::id()) {
            abort(403);
        }

        return view('payment.failed', compact('order'));
    }

    /**
     * API: Get districts for AJAX
     */
    public function getDistricts()
    {
        $districts = District::select('id', 'name', 'province')->orderBy('name')->get();
        return response()->json($districts);
    }

    /**
     * API: Get offices by district
     */
    public function getOfficesByDistrict(District $district)
    {
        $offices = $district->offices()->where('is_active', true)->get();
        return response()->json($offices);
    }

    /**
     * API: Calculate price
     */
    public function calculatePrice(Request $request)
    {
        $request->validate([
            'weight' => 'required|numeric|min:0.1',
            'delivery_type' => 'required|in:standard,express,same_day',
            'vehicle_type' => 'required|in:bike,car,van,truck',
            'pickup_district_id' => 'nullable|exists:districts,id',
            'drop_district_id' => 'nullable|exists:districts,id',
        ]);

        $price = Order::calculatePrice(
            $request->weight,
            $request->delivery_type,
            $request->vehicle_type,
            $request->pickup_district_id,
            $request->drop_district_id
        );

        return response()->json(['price' => $price]);
    }
}
