<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\District;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class HomeController extends Controller
{
    /**
     * Show the home page
     */
    public function index()
    {
        return view('home');
    }

    /**
     * Show the tracking page
     */
    public function tracking()
    {
        return view('tracking.index');
    }

    /**
     * Search for order by tracking code
     */
    public function trackingSearch(Request $request)
    {
        $request->validate([
            'tracking_code' => 'required|string'
        ]);

        $order = Order::where('tracking_code', $request->tracking_code)
                     ->with(['pickupDistrict', 'dropDistrict', 'assignedOffice', 'assignedAgent.user'])
                     ->first();

        if (!$order) {
            return back()->with('error', 'Order not found. Please check your tracking code.');
        }

        return view('tracking.result', compact('order'));
    }

    /**
     * Show user profile
     */
    public function profile()
    {
        $user = Auth::user();
        return view('profile.index', compact('user'));
    }

    /**
     * Update user profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users')->ignore($user->id)
            ],
            'phone' => [
                'nullable',
                'string',
                Rule::unique('users')->ignore($user->id)
            ],
            'current_password' => 'nullable|string',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        // Check current password if new password is provided
        if ($request->filled('password')) {
            if (!$request->filled('current_password') || !Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => 'Current password is incorrect.']);
            }
        }

        // Update user data
        $userData = [
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
        ];

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->password);
        }

        $user->update($userData);

        return back()->with('success', 'Profile updated successfully!');
    }
}
