<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Order;
use App\Models\District;
use App\Models\User;

class MobileController extends Controller
{
    /**
     * Check if the request is from a mobile device
     */
    public function isMobile(Request $request)
    {
        $userAgent = $request->header('User-Agent');
        
        // Check for mobile user agents
        $mobileAgents = [
            'Mobile', 'Android', 'iPhone', 'iPad', 'iPod', 'BlackBerry', 
            'Windows Phone', 'Opera Mini', 'IEMobile', 'Mobile Safari'
        ];
        
        foreach ($mobileAgents as $agent) {
            if (stripos($userAgent, $agent) !== false) {
                return true;
            }
        }
        
        // Check for mobile-specific headers
        if ($request->header('X-Wap-Profile') || 
            $request->header('Profile') || 
            $request->header('X-OperaMini-Features') || 
            $request->header('UA-pixels')) {
            return true;
        }
        
        // Check for mobile viewport
        if ($request->has('mobile') || $request->get('view') === 'mobile') {
            return true;
        }
        
        return false;
    }
    
    /**
     * Mobile home page
     */
    public function home(Request $request)
    {
        // Get recent orders for authenticated users
        $recentOrders = null;
        if (Auth::check()) {
            $recentOrders = Auth::user()->orders()
                ->with(['pickupDistrict', 'dropDistrict'])
                ->latest()
                ->take(3)
                ->get();
        }
        
        return view('mobile.home', compact('recentOrders'));
    }
    
    /**
     * Mobile login page
     */
    public function showLoginForm()
    {
        return view('mobile.auth.login');
    }
    
    /**
     * Mobile profile page
     */
    public function profile()
    {
        if (!Auth::check()) {
            return redirect()->route('mobile.login');
        }
        
        $user = Auth::user();
        $recentOrders = $user->orders()
            ->with(['pickupDistrict', 'dropDistrict'])
            ->latest()
            ->take(5)
            ->get();
            
        return view('mobile.profile.index', compact('user', 'recentOrders'));
    }
    
    /**
     * Mobile orders page
     */
    public function orders(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('mobile.login');
        }
        
        $query = Auth::user()->orders()->with(['pickupDistrict', 'dropDistrict']);
        
        // Filter by status if provided
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        
        // Search by tracking code
        if ($request->has('search') && $request->search != '') {
            $query->where('tracking_code', 'like', '%' . $request->search . '%');
        }
        
        $orders = $query->latest()->paginate(10);
        
        return view('mobile.orders.index', compact('orders'));
    }
    
    /**
     * Mobile order details
     */
    public function showOrder($orderId)
    {
        $order = Order::with(['pickupDistrict', 'dropDistrict', 'assignedAgent.user'])->findOrFail($orderId);
        
        // Check if user owns this order or is admin
        if (!Auth::check() || (Auth::user()->id !== $order->user_id && !Auth::user()->isAdmin())) {
            abort(403, 'Unauthorized access to order details.');
        }
        
        return view('mobile.orders.show', compact('order'));
    }
    
    /**
     * Mobile tracking page
     */
    public function tracking()
    {
        return view('mobile.tracking.index');
    }
    
    /**
     * Mobile tracking search
     */
    public function trackingSearch(Request $request)
    {
        $request->validate([
            'tracking_code' => 'required|string|max:50'
        ]);
        
        $order = Order::where('tracking_code', $request->tracking_code)
            ->with(['pickupDistrict', 'dropDistrict', 'assignedAgent.user', 'user'])
            ->first();
            
        if (!$order) {
            return back()->withErrors([
                'tracking_code' => 'No order found with this tracking code.'
            ])->withInput();
        }
        
        return view('mobile.tracking.result', compact('order'));
    }
    
    /**
     * Mobile courier booking page
     */
    public function createCourier()
    {
        if (!Auth::check()) {
            return redirect()->route('mobile.login');
        }
        
        $districts = District::orderBy('name')
            ->get()
            ->groupBy('province');
            
        return view('mobile.courier.create', compact('districts'));
    }
    
    /**
     * Store mobile courier booking
     */
    public function storeCourier(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('mobile.login');
        }
        
        $request->validate([
            'pickup_district_id' => 'required|exists:districts,id',
            'drop_district_id' => 'required|exists:districts,id|different:pickup_district_id',
            'pickup_address' => 'required|string|max:500',
            'drop_address' => 'required|string|max:500',
            'recipient_name' => 'required|string|max:100',
            'recipient_phone' => 'required|string|max:20',
            'package_type' => 'required|in:document,parcel,fragile',
            'weight' => 'required|numeric|min:0.1|max:50',
            'payment_method' => 'required|in:cod,jazzcash,easypaisa',
            'special_instructions' => 'nullable|string|max:1000'
        ]);
        
        // Calculate price based on distance and weight
        $basePrice = 200; // Base price
        $weightPrice = $request->weight * 50; // Per kg price
        $totalPrice = $basePrice + $weightPrice;
        
        // Generate unique tracking code
        do {
            $trackingCode = 'TCS' . date('Ymd') . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT);
        } while (Order::where('tracking_code', $trackingCode)->exists());
        
        // Create order
        $order = Order::create([
            'user_id' => Auth::id(),
            'tracking_code' => $trackingCode,
            'pickup_district_id' => $request->pickup_district_id,
            'drop_district_id' => $request->drop_district_id,
            'pickup_address' => $request->pickup_address,
            'drop_address' => $request->drop_address,
            'recipient_name' => $request->recipient_name,
            'recipient_phone' => $request->recipient_phone,
            'package_type' => $request->package_type,
            'weight' => $request->weight,
            'base_price' => $basePrice,
            'final_price' => $totalPrice,
            'payment_method' => $request->payment_method,
            'payment_status' => $request->payment_method === 'cod' ? 'pending' : 'paid',
            'special_instructions' => $request->special_instructions,
            'status' => 'created'
        ]);
        
        return redirect()->route('mobile.orders.show', $order)
            ->with('success', 'Your courier booking has been confirmed! Tracking Code: ' . $trackingCode);
    }
    
    /**
     * Update user profile from mobile
     */
    public function updateProfile(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('mobile.login');
        }
        
        $user = Auth::user();
        
        // Check if this is a password change request
        if ($request->has('current_password')) {
            $request->validate([
                'current_password' => 'required|string',
                'password' => 'required|string|min:8|confirmed',
            ]);
            
            if (!\Hash::check($request->current_password, $user->password)) {
                return back()->withErrors([
                    'current_password' => 'Current password is incorrect.'
                ]);
            }
            
            $user->update([
                'password' => \Hash::make($request->password)
            ]);
            
            return back()->with('success', 'Password updated successfully!');
        }
        
        // Regular profile update
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
        ]);
        
        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
        ]);
        
        return back()->with('success', 'Profile updated successfully!');
    }
    
    /**
     * Get mobile app manifest
     */
    public function manifest()
    {
        $manifest = [
            'name' => 'TCS Courier Mobile App',
            'short_name' => 'TCS Courier',
            'description' => 'Fast and reliable courier service mobile app',
            'start_url' => route('mobile.home'),
            'display' => 'standalone',
            'background_color' => '#4e73df',
            'theme_color' => '#4e73df',
            'orientation' => 'portrait',
            'icons' => [
                [
                    'src' => '/favicon.ico',
                    'sizes' => '192x192',
                    'type' => 'image/png'
                ],
                [
                    'src' => '/favicon.ico',
                    'sizes' => '512x512',
                    'type' => 'image/png'
                ]
            ]
        ];
        
        return response()->json($manifest)
            ->header('Content-Type', 'application/manifest+json');
    }
}