<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check if user is authenticated
        if (!Auth::check()) {
            // Store the intended URL for redirect after login
            session(['url.intended' => $request->url()]);
            
            // Redirect to admin login with message
            return redirect()->route('admin.login')
                ->with('error', 'Please login to access the admin portal.');
        }

        // Check if authenticated user is an admin
        if (!Auth::user()->isAdmin()) {
            // Log unauthorized access attempt
            \Log::warning('Unauthorized admin access attempt', [
                'user_id' => Auth::id(),
                'user_email' => Auth::user()->email,
                'user_role' => Auth::user()->role,
                'requested_url' => $request->url(),
                'ip_address' => $request->ip(),
                'timestamp' => now()
            ]);

            // Logout the user and redirect to admin login
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            
            return redirect()->route('admin.login')
                ->with('error', 'Access denied. This area is restricted to administrators only.');
        }

        // Check if admin account is active
        if (!Auth::user()->is_active) {
            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();
            
            return redirect()->route('admin.login')
                ->with('error', 'Your admin account has been deactivated. Please contact the system administrator.');
        }

        return $next($request);
    }
}
