<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Session;

class SetLocale
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Available languages
        $availableLanguages = ['en', 'ur'];
        
        // Get language from session, URL parameter, or browser preference
        $locale = $this->getPreferredLanguage($request, $availableLanguages);
        
        // Set application locale
        App::setLocale($locale);
        
        // Store in session for future requests
        Session::put('locale', $locale);
        
        return $next($request);
    }
    
    /**
     * Get the preferred language based on various sources
     */
    private function getPreferredLanguage(Request $request, array $availableLanguages): string
    {
        // 1. Check URL parameter (for mobile=1&lang=ur)
        if ($request->has('lang') && in_array($request->get('lang'), $availableLanguages)) {
            return $request->get('lang');
        }
        
        // 2. Check session
        if (Session::has('locale') && in_array(Session::get('locale'), $availableLanguages)) {
            return Session::get('locale');
        }
        
        // 3. Check user preference (if authenticated)
        if (auth()->check() && auth()->user()->language && in_array(auth()->user()->language, $availableLanguages)) {
            return auth()->user()->language;
        }
        
        // 4. Check browser language preference
        $browserLanguage = $this->getBrowserLanguage($request, $availableLanguages);
        if ($browserLanguage) {
            return $browserLanguage;
        }
        
        // 5. Default to English
        return 'en';
    }
    
    /**
     * Get browser language preference
     */
    private function getBrowserLanguage(Request $request, array $availableLanguages): ?string
    {
        $acceptLanguage = $request->header('Accept-Language');
        
        if (!$acceptLanguage) {
            return null;
        }
        
        // Parse Accept-Language header
        $languages = [];
        foreach (explode(',', $acceptLanguage) as $lang) {
            $parts = explode(';', trim($lang));
            $code = trim($parts[0]);
            $priority = 1.0;
            
            if (isset($parts[1]) && strpos($parts[1], 'q=') === 0) {
                $priority = (float) substr($parts[1], 2);
            }
            
            // Extract language code (e.g., 'ur-PK' -> 'ur')
            $langCode = substr($code, 0, 2);
            
            if (in_array($langCode, $availableLanguages)) {
                $languages[$langCode] = $priority;
            }
        }
        
        // Sort by priority and return the highest
        if (!empty($languages)) {
            arsort($languages);
            return array_key_first($languages);
        }
        
        return null;
    }
}