<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DistrictPricing extends Model
{
    use HasFactory;

    protected $fillable = [
        'pickup_district_id',
        'drop_district_id',
        'service_type',
        'base_rate',
        'per_kg_rate',
        'minimum_charge',
        'maximum_charge',
        'estimated_days',
        'is_active',
        'notes',
    ];

    protected $casts = [
        'base_rate' => 'decimal:2',
        'per_kg_rate' => 'decimal:2',
        'minimum_charge' => 'decimal:2',
        'maximum_charge' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    /**
     * Service type constants
     */
    const SERVICE_STANDARD = 'standard';
    const SERVICE_EXPRESS = 'express';
    const SERVICE_OVERNIGHT = 'overnight';

    /**
     * Get available service types
     */
    public static function getServiceTypes()
    {
        return [
            self::SERVICE_STANDARD => 'Standard Delivery',
            self::SERVICE_EXPRESS => 'Express Delivery',
            self::SERVICE_OVERNIGHT => 'Overnight Delivery',
        ];
    }

    /**
     * Get the pickup district
     */
    public function pickupDistrict()
    {
        return $this->belongsTo(District::class, 'pickup_district_id');
    }

    /**
     * Get the drop district
     */
    public function dropDistrict()
    {
        return $this->belongsTo(District::class, 'drop_district_id');
    }

    /**
     * Calculate total price for given weight
     */
    public function calculatePrice($weight = 1)
    {
        $price = $this->base_rate + ($this->per_kg_rate * $weight);
        
        // Apply minimum charge
        if ($price < $this->minimum_charge) {
            $price = $this->minimum_charge;
        }
        
        // Apply maximum charge if set
        if ($this->maximum_charge && $price > $this->maximum_charge) {
            $price = $this->maximum_charge;
        }
        
        return round($price, 2);
    }

    /**
     * Scope for active pricing
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Get pricing for specific route and service
     */
    public static function getPricing($pickupDistrictId, $dropDistrictId, $serviceType = self::SERVICE_STANDARD)
    {
        return self::where('pickup_district_id', $pickupDistrictId)
                   ->where('drop_district_id', $dropDistrictId)
                   ->where('service_type', $serviceType)
                   ->where('is_active', true)
                   ->first();
    }
}
