<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'tracking_code',
        'user_id',
        'pickup_district_id',
        'drop_district_id',
        'assigned_office_id',
        'assigned_agent_id',
        'sender_name',
        'sender_phone',
        'sender_address',
        'receiver_name',
        'receiver_phone',
        'receiver_address',
        'weight',
        'size',
        'category',
        'description',
        'delivery_type',
        'vehicle_type',
        'estimated_price',
        'final_price',
        'payment_method',
        'payment_status',
        'payment_reference',
        'status',
        'picked_at',
        'delivered_at',
        'notes',
    ];

    protected $casts = [
        'weight' => 'decimal:2',
        'estimated_price' => 'decimal:2',
        'final_price' => 'decimal:2',
        'picked_at' => 'datetime',
        'delivered_at' => 'datetime',
    ];

    /**
     * Get the user that owns the order.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the pickup district.
     */
    public function pickupDistrict()
    {
        return $this->belongsTo(District::class, 'pickup_district_id');
    }

    /**
     * Get the drop district.
     */
    public function dropDistrict()
    {
        return $this->belongsTo(District::class, 'drop_district_id');
    }

    /**
     * Get the assigned office.
     */
    public function assignedOffice()
    {
        return $this->belongsTo(Office::class, 'assigned_office_id');
    }

    /**
     * Get the assigned agent.
     */
    public function assignedAgent()
    {
        return $this->belongsTo(Agent::class, 'assigned_agent_id');
    }

    /**
     * Generate unique tracking code
     */
    public static function generateTrackingCode()
    {
        do {
            $code = 'TCS' . date('Ymd') . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        } while (self::where('tracking_code', $code)->exists());
        
        return $code;
    }

    /**
     * Calculate estimated price based on districts, weight, and delivery type
     */
    public static function calculatePrice($weight, $deliveryType, $vehicleType = null, $pickupDistrictId = null, $dropDistrictId = null)
    {
        // If district IDs are provided, try to get district-based pricing
        if ($pickupDistrictId && $dropDistrictId) {
            $districtPricing = DistrictPricing::getPricing($pickupDistrictId, $dropDistrictId, $deliveryType);
            
            if ($districtPricing) {
                return $districtPricing->calculatePrice($weight);
            }
        }
        
        // Fallback to old pricing system if no district pricing found
        $basePrice = 100; // Base price in PKR
        $weightMultiplier = $weight * 10;
        
        $deliveryMultiplier = [
            'standard' => 1,
            'express' => 1.5,
            'same_day' => 2
        ];
        
        $vehicleMultiplier = [
            'bike' => 1,
            'car' => 1.2,
            'van' => 1.5,
            'truck' => 2
        ];
        
        $price = $basePrice + $weightMultiplier;
        $price *= $deliveryMultiplier[$deliveryType] ?? 1;
        $price *= $vehicleMultiplier[$vehicleType] ?? 1;
        
        return round($price, 2);
    }

    /**
     * Get status badge color
     */
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'created' => 'secondary',
            'picked' => 'warning',
            'in_transit' => 'info',
            'delivered' => 'success',
            'cancelled' => 'danger'
        ];
        
        return $badges[$this->status] ?? 'secondary';
    }
}
