<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\District;
use App\Models\DistrictPricing;

class DistrictPricingSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Get all districts
        $districts = District::all();
        
        // Define base pricing structure
        $pricingStructure = [
            // Same province pricing (cheaper)
            'same_province' => [
                'standard' => ['base_rate' => 150, 'per_kg_rate' => 25, 'minimum_charge' => 200, 'estimated_days' => 2],
                'express' => ['base_rate' => 200, 'per_kg_rate' => 35, 'minimum_charge' => 300, 'estimated_days' => 1],
                'overnight' => ['base_rate' => 300, 'per_kg_rate' => 50, 'minimum_charge' => 450, 'estimated_days' => 1],
            ],
            // Different province pricing (more expensive)
            'different_province' => [
                'standard' => ['base_rate' => 250, 'per_kg_rate' => 40, 'minimum_charge' => 350, 'estimated_days' => 3],
                'express' => ['base_rate' => 350, 'per_kg_rate' => 55, 'minimum_charge' => 500, 'estimated_days' => 2],
                'overnight' => ['base_rate' => 500, 'per_kg_rate' => 75, 'minimum_charge' => 700, 'estimated_days' => 1],
            ],
            // Same district pricing (cheapest)
            'same_district' => [
                'standard' => ['base_rate' => 100, 'per_kg_rate' => 15, 'minimum_charge' => 150, 'estimated_days' => 1],
                'express' => ['base_rate' => 150, 'per_kg_rate' => 25, 'minimum_charge' => 200, 'estimated_days' => 1],
                'overnight' => ['base_rate' => 200, 'per_kg_rate' => 35, 'minimum_charge' => 300, 'estimated_days' => 1],
            ],
        ];

        // Create pricing for all district combinations
        foreach ($districts as $pickupDistrict) {
            foreach ($districts as $dropDistrict) {
                // Determine pricing category
                if ($pickupDistrict->id === $dropDistrict->id) {
                    $category = 'same_district';
                } elseif ($pickupDistrict->province === $dropDistrict->province) {
                    $category = 'same_province';
                } else {
                    $category = 'different_province';
                }

                // Create pricing for each service type
                foreach (['standard', 'express', 'overnight'] as $serviceType) {
                    $pricing = $pricingStructure[$category][$serviceType];
                    
                    // Check if pricing already exists
                    $existing = DistrictPricing::where('pickup_district_id', $pickupDistrict->id)
                        ->where('drop_district_id', $dropDistrict->id)
                        ->where('service_type', $serviceType)
                        ->first();

                    if (!$existing) {
                        DistrictPricing::create([
                            'pickup_district_id' => $pickupDistrict->id,
                            'drop_district_id' => $dropDistrict->id,
                            'service_type' => $serviceType,
                            'base_rate' => $pricing['base_rate'],
                            'per_kg_rate' => $pricing['per_kg_rate'],
                            'minimum_charge' => $pricing['minimum_charge'],
                            'maximum_charge' => null, // No maximum limit
                            'estimated_days' => $pricing['estimated_days'],
                            'is_active' => true,
                            'notes' => "Auto-generated pricing for {$pickupDistrict->name} to {$dropDistrict->name} ({$serviceType})",
                        ]);
                    }
                }
            }
        }

        $this->command->info('District pricing seeded successfully!');
    }
}