@extends('layouts.app')

@section('title', 'Manage Districts')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-geo-alt text-primary"></i> District Management
            </h1>
            <p class="text-muted mb-0">Manage service areas and districts</p>
        </div>
        <div>
            <a href="{{ route('admin.dashboard') }}" class="btn btn-outline-secondary me-2">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
            <a href="{{ route('admin.districts.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Add New District
            </a>
        </div>
    </div>
    
    <!-- Search & Filters -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-funnel"></i> Search & Filters
            </h6>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('admin.districts') }}">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="{{ request('search') }}" 
                               placeholder="District name or province...">
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="bi bi-search"></i> Search
                        </button>
                        <a href="{{ route('admin.districts') }}" class="btn btn-outline-secondary">
                            <i class="bi bi-x-circle"></i> Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Districts Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-list"></i> Districts List ({{ $districts->total() }} total)
            </h6>
            <div class="dropdown">
                <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="bi bi-download"></i> Export
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-excel"></i> Export to Excel</a></li>
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-pdf"></i> Export to PDF</a></li>
                </ul>
            </div>
        </div>
        <div class="card-body">
            @if($districts->count() > 0)
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th>District Info</th>
                                <th>Province</th>
                                <th>Statistics</th>
                                <th>Status</th>
                                <th>Created Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($districts as $district)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="district-icon bg-primary text-white me-3">
                                                <i class="bi bi-geo-alt"></i>
                                            </div>
                                            <div>
                                                <strong>{{ $district->name }}</strong>
                                                <br><small class="text-muted">ID: {{ $district->id }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge bg-info fs-6">{{ $district->province }}</span>
                                    </td>
                                    <td>
                                        <div class="row text-center">
                                            <div class="col-4">
                                                <div class="small">
                                                    <div class="h6 text-primary mb-0">{{ $district->pickup_orders_count }}</div>
                                                    <small class="text-muted">Pickups</small>
                                                </div>
                                            </div>
                                            <div class="col-4">
                                                <div class="small">
                                                    <div class="h6 text-success mb-0">{{ $district->drop_orders_count }}</div>
                                                    <small class="text-muted">Drops</small>
                                                </div>
                                            </div>
                                            <div class="col-4">
                                                <div class="small">
                                                    <div class="h6 text-info mb-0">{{ $district->offices_count }}</div>
                                                    <small class="text-muted">Offices</small>
                                                </div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="text-center">
                                            @if($district->is_active)
                                                <span class="badge bg-success">Active</span>
                                                <br><small class="text-muted">Available</small>
                                            @else
                                                <span class="badge bg-danger">Inactive</span>
                                                <br><small class="text-muted">Unavailable</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            {{ $district->created_at->format('M d, Y') }}
                                            <br>{{ $district->created_at->format('h:i A') }}
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('admin.districts.edit', $district) }}" 
                                               class="btn btn-sm btn-outline-primary" title="Edit District">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-outline-info" 
                                                    data-bs-toggle="modal" data-bs-target="#viewModal{{ $district->id }}" 
                                                    title="View Details">
                                                <i class="bi bi-eye"></i>
                                            </button>
                                            @if($district->pickup_orders_count == 0 && $district->drop_orders_count == 0 && $district->offices_count == 0)
                                                <button type="button" class="btn btn-sm btn-outline-danger" 
                                                        onclick="deleteDistrict({{ $district->id }})" 
                                                        title="Delete District">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                
                                <!-- View Details Modal -->
                                <div class="modal fade" id="viewModal{{ $district->id }}" tabindex="-1">
                                    <div class="modal-dialog modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">District Details - {{ $district->name }}</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                            </div>
                                            <div class="modal-body">
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <h6 class="text-primary mb-3">Basic Information</h6>
                                                        <div class="mb-2"><strong>Name:</strong> {{ $district->name }}</div>
                                                        <div class="mb-2"><strong>Province:</strong> {{ $district->province }}</div>
                                                        <div class="mb-2"><strong>Status:</strong> 
                                                            <span class="badge bg-{{ $district->is_active ? 'success' : 'danger' }}">
                                                                {{ $district->is_active ? 'Active' : 'Inactive' }}
                                                            </span>
                                                        </div>
                                                        <div class="mb-2"><strong>Created:</strong> {{ $district->created_at->format('M d, Y h:i A') }}</div>
                                                        <div class="mb-2"><strong>Last Updated:</strong> {{ $district->updated_at->format('M d, Y h:i A') }}</div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <h6 class="text-success mb-3">Service Statistics</h6>
                                                        <div class="mb-2"><strong>Pickup Orders:</strong> {{ $district->pickup_orders_count }}</div>
                                                        <div class="mb-2"><strong>Drop Orders:</strong> {{ $district->drop_orders_count }}</div>
                                                        <div class="mb-2"><strong>Total Orders:</strong> {{ $district->pickup_orders_count + $district->drop_orders_count }}</div>
                                                        <div class="mb-2"><strong>Offices:</strong> {{ $district->offices_count }}</div>
                                                    </div>
                                                </div>
                                                
                                                <!-- Activity Chart -->
                                                <hr>
                                                <h6 class="text-info mb-3">Activity Overview</h6>
                                                <div class="row text-center">
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-primary mb-1">{{ $district->pickup_orders_count }}</div>
                                                            <small class="text-muted">Pickup Orders</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-success mb-1">{{ $district->drop_orders_count }}</div>
                                                            <small class="text-muted">Drop Orders</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-info mb-1">{{ $district->offices_count }}</div>
                                                            <small class="text-muted">Service Offices</small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                <a href="{{ route('admin.districts.edit', $district) }}" class="btn btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit District
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-4">
                    {{ $districts->appends(request()->query())->links() }}
                </div>
            @else
                <div class="text-center py-5">
                    <i class="bi bi-geo-alt text-muted" style="font-size: 4rem;"></i>
                    <h4 class="text-muted mt-3">No Districts Found</h4>
                    <p class="text-muted">No districts match your current search criteria.</p>
                    <a href="{{ route('admin.districts.create') }}" class="btn btn-primary">
                        <i class="bi bi-plus-circle"></i> Add First District
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    .district-icon {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
    }
    
    .table th {
        border-top: none;
        font-weight: 600;
        font-size: 0.85rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table td {
        vertical-align: middle;
        font-size: 0.9rem;
    }
    
    .btn-group .btn {
        border-radius: 0.25rem;
        margin-right: 2px;
    }
    
    .badge {
        font-size: 0.75rem;
    }
    
    .card {
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        border: 1px solid #e3e6f0;
    }
    
    .card-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #e3e6f0;
    }
    
    .modal-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #dee2e6;
    }
</style>
@endpush

@push('scripts')
<script>
    function deleteDistrict(districtId) {
        if (confirm('Are you sure you want to delete this district? This action cannot be undone.')) {
            // Create a form and submit it
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/districts/' + districtId;
            
            // Add CSRF token
            const csrfToken = document.createElement('input');
            csrfToken.type = 'hidden';
            csrfToken.name = '_token';
            csrfToken.value = '{{ csrf_token() }}';
            form.appendChild(csrfToken);
            
            // Add method override
            const methodField = document.createElement('input');
            methodField.type = 'hidden';
            methodField.name = '_method';
            methodField.value = 'DELETE';
            form.appendChild(methodField);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Show success/error messages
    @if(session('success'))
        const successToast = document.createElement('div');
        successToast.className = 'alert alert-success alert-dismissible fade show position-fixed';
        successToast.style.top = '20px';
        successToast.style.right = '20px';
        successToast.style.zIndex = '9999';
        successToast.innerHTML = `
            <i class="bi bi-check-circle"></i> {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        document.body.appendChild(successToast);
        
        setTimeout(() => {
            successToast.remove();
        }, 5000);
    @endif
    
    @if(session('error'))
        const errorToast = document.createElement('div');
        errorToast.className = 'alert alert-danger alert-dismissible fade show position-fixed';
        errorToast.style.top = '20px';
        errorToast.style.right = '20px';
        errorToast.style.zIndex = '9999';
        errorToast.innerHTML = `
            <i class="bi bi-exclamation-triangle"></i> {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        document.body.appendChild(errorToast);
        
        setTimeout(() => {
            errorToast.remove();
        }, 5000);
    @endif
</script>
@endpush