@extends('layouts.app')

@section('title', 'Create Office')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-building-add text-primary"></i> Create New Office
            </h1>
            <p class="text-muted mb-0">Add a new courier office to the system</p>
        </div>
        <div>
            <a href="{{ route('admin.offices') }}" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Offices
            </a>
        </div>
    </div>
    
    <!-- Create Office Form -->
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="card shadow">
                <div class="card-header">
                    <h5 class="mb-0"><i class="bi bi-building"></i> Office Information</h5>
                </div>
                <div class="card-body">
                    <form method="POST" action="{{ route('admin.offices.store') }}">
                        @csrf
                        
                        <!-- Office Name -->
                        <div class="mb-4">
                            <label for="name" class="form-label">Office Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                   id="name" name="name" value="{{ old('name') }}" 
                                   placeholder="e.g., TCS Lahore Main Office" required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">Enter a descriptive name for the office</small>
                        </div>
                        
                        <!-- District Selection -->
                        <div class="mb-4">
                            <label for="district_id" class="form-label">District <span class="text-danger">*</span></label>
                            <select class="form-select @error('district_id') is-invalid @enderror" id="district_id" name="district_id" required>
                                <option value="">Select district...</option>
                                @foreach($districts as $district)
                                    <option value="{{ $district->id }}" {{ old('district_id') == $district->id ? 'selected' : '' }}>
                                        {{ $district->name }}, {{ $district->province }}
                                    </option>
                                @endforeach
                            </select>
                            @error('district_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">Select the district where this office is located</small>
                        </div>
                        
                        <!-- Address -->
                        <div class="mb-4">
                            <label for="address" class="form-label">Complete Address <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('address') is-invalid @enderror" 
                                      id="address" name="address" rows="3" 
                                      placeholder="Enter complete address with landmarks" required>{{ old('address') }}</textarea>
                            @error('address')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="row">
                            <!-- Phone -->
                            <div class="col-md-6 mb-3">
                                <label for="phone" class="form-label">Phone Number <span class="text-danger">*</span></label>
                                <input type="tel" class="form-control @error('phone') is-invalid @enderror" 
                                       id="phone" name="phone" value="{{ old('phone') }}" 
                                       placeholder="+92-42-1234567" required>
                                @error('phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            
                            <!-- Email -->
                            <div class="col-md-6 mb-3">
                                <label for="email" class="form-label">Email Address</label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror" 
                                       id="email" name="email" value="{{ old('email') }}" 
                                       placeholder="office@tcscourier.com">
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="form-text text-muted">Optional - for official correspondence</small>
                            </div>
                        </div>
                        
                        <!-- Manager Information -->
                        <div class="mb-4">
                            <h6 class="text-primary mb-3">Manager Information</h6>
                            <div class="row">
                                <div class="col-md-12">
                                    <label for="manager_name" class="form-label">Manager Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('manager_name') is-invalid @enderror" 
                                           id="manager_name" name="manager_name" value="{{ old('manager_name') }}" 
                                           placeholder="Full name of the office manager" required>
                                    @error('manager_name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>
                        
                        <!-- Office Status -->
                        <div class="mb-4">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                       {{ old('is_active', true) ? 'checked' : '' }}>
                                <label class="form-check-label" for="is_active">
                                    <strong>Active Office</strong>
                                    <small class="text-muted d-block">Office is operational and can handle orders</small>
                                </label>
                            </div>
                        </div>
                        
                        <!-- Additional Information -->
                        <div class="alert alert-info" role="alert">
                            <h6 class="alert-heading">
                                <i class="bi bi-info-circle"></i> Important Information
                            </h6>
                            <ul class="mb-0 small">
                                <li>Make sure the address is accurate for GPS navigation</li>
                                <li>Phone number will be used for customer inquiries</li>
                                <li>Manager will be responsible for office operations</li>
                                <li>Active offices can receive order assignments</li>
                            </ul>
                        </div>
                        
                        <!-- Submit Buttons -->
                        <div class="d-flex justify-content-end gap-2">
                            <a href="{{ route('admin.offices') }}" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Create Office
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    .card {
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        border: 1px solid #e3e6f0;
    }
    
    .card-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #e3e6f0;
    }
    
    .form-label {
        font-weight: 600;
        color: #5a5c69;
    }
    
    .text-danger {
        font-size: 0.9rem;
    }
    
    .form-text {
        font-size: 0.8rem;
    }
    
    .btn {
        font-weight: 500;
    }
    
    .alert {
        border-left: 4px solid #17a2b8;
    }
    
    .alert-heading {
        font-size: 1rem;
        margin-bottom: 0.5rem;
    }
</style>
@endpush

@push('scripts')
<script>
    // Auto-format phone number
    document.getElementById('phone').addEventListener('input', function(e) {
        let value = e.target.value.replace(/\D/g, '');
        if (value.length > 0) {
            if (value.startsWith('92')) {
                value = '+' + value;
            } else if (!value.startsWith('+92')) {
                value = '+92' + value;
            }
        }
        e.target.value = value;
    });
    
    // Auto-generate email based on office name and district
    function generateEmail() {
        const officeName = document.getElementById('name').value;
        const districtSelect = document.getElementById('district_id');
        const emailField = document.getElementById('email');
        
        if (officeName && districtSelect.value && !emailField.value) {
            const districtName = districtSelect.options[districtSelect.selectedIndex].text.split(',')[0];
            const cleanOfficeName = officeName.toLowerCase().replace(/[^a-z0-9]/g, '');
            const cleanDistrictName = districtName.toLowerCase().replace(/[^a-z0-9]/g, '');
            emailField.value = cleanOfficeName + '.' + cleanDistrictName + '@tcscourier.com';
        }
    }
    
    document.getElementById('name').addEventListener('blur', generateEmail);
    document.getElementById('district_id').addEventListener('change', generateEmail);
    
    // Form validation
    document.querySelector('form').addEventListener('submit', function(e) {
        const requiredFields = ['name', 'district_id', 'address', 'phone', 'manager_name'];
        let isValid = true;
        
        requiredFields.forEach(field => {
            const input = document.getElementById(field);
            if (!input.value.trim()) {
                input.classList.add('is-invalid');
                isValid = false;
            } else {
                input.classList.remove('is-invalid');
            }
        });
        
        if (!isValid) {
            e.preventDefault();
            alert('Please fill in all required fields.');
        }
    });
    
    // Real-time validation feedback
    document.querySelectorAll('input, select, textarea').forEach(field => {
        field.addEventListener('input', function() {
            if (this.value.trim()) {
                this.classList.remove('is-invalid');
                this.classList.add('is-valid');
            }
        });
    });
</script>
@endpush