@extends('layouts.app')

@section('title', 'Edit Office')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">Edit Office</h1>
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb mb-0">
                    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
                    <li class="breadcrumb-item"><a href="{{ route('admin.offices') }}">Offices</a></li>
                    <li class="breadcrumb-item active">Edit Office</li>
                </ol>
            </nav>
        </div>
        <a href="{{ route('admin.offices') }}" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Back to Offices
        </a>
    </div>

    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow">
                <div class="card-header">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="bi bi-building"></i> Office Information
                    </h6>
                </div>
                <div class="card-body">
                    <form action="{{ route('admin.offices.update', $office) }}" method="POST" id="officeForm">
                        @csrf
                        @method('PUT')
                        
                        <!-- Office Name -->
                        <div class="mb-3">
                            <label for="name" class="form-label">Office Name <span class="text-danger">*</span></label>
                            <input type="text" 
                                   class="form-control @error('name') is-invalid @enderror" 
                                   id="name" 
                                   name="name" 
                                   value="{{ old('name', $office->name) }}" 
                                   required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- District -->
                        <div class="mb-3">
                            <label for="district_id" class="form-label">District <span class="text-danger">*</span></label>
                            <select class="form-select @error('district_id') is-invalid @enderror" 
                                    id="district_id" 
                                    name="district_id" 
                                    required>
                                <option value="">Select District</option>
                                @foreach($districts as $district)
                                    <option value="{{ $district->id }}" 
                                            {{ old('district_id', $office->district_id) == $district->id ? 'selected' : '' }}>
                                        {{ $district->name }} ({{ $district->province }})
                                    </option>
                                @endforeach
                            </select>
                            @error('district_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Address -->
                        <div class="mb-3">
                            <label for="address" class="form-label">Address <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('address') is-invalid @enderror" 
                                      id="address" 
                                      name="address" 
                                      rows="3" 
                                      required>{{ old('address', $office->address) }}</textarea>
                            @error('address')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Phone -->
                        <div class="mb-3">
                            <label for="phone" class="form-label">Phone Number <span class="text-danger">*</span></label>
                            <input type="tel" 
                                   class="form-control @error('phone') is-invalid @enderror" 
                                   id="phone" 
                                   name="phone" 
                                   value="{{ old('phone', $office->phone) }}" 
                                   placeholder="e.g., 011-2345678"
                                   required>
                            @error('phone')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <div class="form-text">Format: 011-2345678 or 077-1234567</div>
                        </div>

                        <!-- Email -->
                        <div class="mb-3">
                            <label for="email" class="form-label">Email Address</label>
                            <input type="email" 
                                   class="form-control @error('email') is-invalid @enderror" 
                                   id="email" 
                                   name="email" 
                                   value="{{ old('email', $office->email) }}" 
                                   placeholder="office@example.com">
                            @error('email')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <div class="form-text">Optional - Will be auto-generated if not provided</div>
                        </div>

                        <!-- Manager Information -->
                        <div class="card mt-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Manager Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <label for="manager_name" class="form-label">Manager Name <span class="text-danger">*</span></label>
                                    <input type="text" 
                                           class="form-control @error('manager_name') is-invalid @enderror" 
                                           id="manager_name" 
                                           name="manager_name" 
                                           value="{{ old('manager_name', $office->manager_name) }}" 
                                           required>
                                    @error('manager_name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Office Status -->
                        <div class="card mt-4">
                            <div class="card-header bg-light">
                                <h6 class="mb-0">Office Status</h6>
                            </div>
                            <div class="card-body">
                                <div class="form-check">
                                    <input class="form-check-input" 
                                           type="checkbox" 
                                           id="is_active" 
                                           name="is_active" 
                                           value="1"
                                           {{ old('is_active', $office->is_active) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="is_active">
                                        <strong>Active Office</strong>
                                    </label>
                                    <div class="form-text">
                                        Active offices can accept new orders and be assigned to agents. 
                                        Inactive offices will not appear in selection lists.
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Additional Information -->
                        <div class="alert alert-info mt-4">
                            <i class="bi bi-info-circle"></i>
                            <strong>Note:</strong> Changes to office information will affect all associated agents and future orders. 
                            Make sure all information is accurate before saving.
                        </div>

                        <!-- Submit Buttons -->
                        <div class="d-flex justify-content-end gap-2 mt-4">
                            <a href="{{ route('admin.offices') }}" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Update Office
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Office Statistics -->
        <div class="col-lg-4">
            <div class="card shadow">
                <div class="card-header">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="bi bi-graph-up"></i> Office Statistics
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row text-center">
                        <div class="col-6">
                            <div class="border-end">
                                <h4 class="text-primary mb-1">{{ $office->agents()->count() }}</h4>
                                <small class="text-muted">Active Agents</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <h4 class="text-success mb-1">{{ $office->orders()->count() }}</h4>
                            <small class="text-muted">Total Orders</small>
                        </div>
                    </div>
                    
                    <hr>
                    
                    <div class="mb-3">
                        <small class="text-muted d-block">Office Status</small>
                        @if($office->is_active)
                            <span class="badge bg-success">Active</span>
                        @else
                            <span class="badge bg-secondary">Inactive</span>
                        @endif
                    </div>
                    
                    <div class="mb-3">
                        <small class="text-muted d-block">Created</small>
                        <span class="text-dark">{{ $office->created_at->format('M d, Y') }}</span>
                    </div>
                    
                    <div class="mb-3">
                        <small class="text-muted d-block">Last Updated</small>
                        <span class="text-dark">{{ $office->updated_at->format('M d, Y') }}</span>
                    </div>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div class="card shadow mt-4">
                <div class="card-header">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="bi bi-lightning"></i> Quick Actions
                    </h6>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <a href="{{ route('admin.agents.create') }}?office_id={{ $office->id }}" class="btn btn-outline-primary btn-sm">
                            <i class="bi bi-person-plus"></i> Add Agent to Office
                        </a>
                        <a href="{{ route('admin.offices') }}?search={{ $office->district->name }}" class="btn btn-outline-info btn-sm">
                            <i class="bi bi-geo-alt"></i> View District Offices
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.form-label {
    font-weight: 600;
    color: #374151;
}

.card {
    border: none;
    border-radius: 10px;
}

.card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 10px 10px 0 0 !important;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #5a6fd8 0%, #6a4190 100%);
    transform: translateY(-1px);
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}

.alert-info {
    background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%);
    border: none;
    border-left: 4px solid #667eea;
}

.badge {
    font-size: 0.75em;
    padding: 0.5em 0.75em;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Phone number formatting
    const phoneInput = document.getElementById('phone');
    phoneInput.addEventListener('input', function(e) {
        let value = e.target.value.replace(/\D/g, '');
        
        if (value.length >= 3) {
            if (value.startsWith('0')) {
                // Format: 011-2345678 or 077-1234567
                if (value.length <= 10) {
                    value = value.replace(/(\d{3})(\d{0,7})/, '$1-$2');
                }
            }
        }
        
        e.target.value = value;
    });

    // Auto-generate email if not provided
    const nameInput = document.getElementById('name');
    const emailInput = document.getElementById('email');
    
    nameInput.addEventListener('blur', function() {
        if (!emailInput.value && nameInput.value) {
            const emailName = nameInput.value.toLowerCase()
                .replace(/[^a-z0-9\s]/g, '')
                .replace(/\s+/g, '.');
            emailInput.value = emailName + '@tcsoffice.lk';
        }
    });

    // Form validation
    const form = document.getElementById('officeForm');
    form.addEventListener('submit', function(e) {
        const requiredFields = ['name', 'district_id', 'address', 'phone', 'manager_name'];
        let isValid = true;

        requiredFields.forEach(function(fieldName) {
            const field = document.getElementById(fieldName);
            if (!field.value.trim()) {
                field.classList.add('is-invalid');
                isValid = false;
            } else {
                field.classList.remove('is-invalid');
            }
        });

        // Phone validation
        const phoneValue = phoneInput.value;
        const phonePattern = /^0\d{2}-\d{7}$/;
        if (phoneValue && !phonePattern.test(phoneValue)) {
            phoneInput.classList.add('is-invalid');
            isValid = false;
        }

        if (!isValid) {
            e.preventDefault();
            alert('Please fill in all required fields correctly.');
        }
    });
});
</script>
@endsection