@extends('layouts.app')

@section('title', 'Manage Offices')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-building text-primary"></i> Office Management
            </h1>
            <p class="text-muted mb-0">Manage courier offices and their locations</p>
        </div>
        <div>
            <a href="{{ route('admin.dashboard') }}" class="btn btn-outline-secondary me-2">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
            <a href="{{ route('admin.offices.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Add New Office
            </a>
        </div>
    </div>
    
    <!-- Search & Filters -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-funnel"></i> Search & Filters
            </h6>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('admin.offices') }}">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="{{ request('search') }}" 
                               placeholder="Office name, address, phone...">
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="bi bi-search"></i> Search
                        </button>
                        <a href="{{ route('admin.offices') }}" class="btn btn-outline-secondary">
                            <i class="bi bi-x-circle"></i> Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Offices Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-list"></i> Offices List ({{ $offices->total() }} total)
            </h6>
            <div class="dropdown">
                <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="bi bi-download"></i> Export
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-excel"></i> Export to Excel</a></li>
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-pdf"></i> Export to PDF</a></li>
                </ul>
            </div>
        </div>
        <div class="card-body">
            @if($offices->count() > 0)
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th>Office Info</th>
                                <th>Location</th>
                                <th>Contact Details</th>
                                <th>Manager</th>
                                <th>Status</th>
                                <th>Created Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($offices as $office)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="office-icon bg-primary text-white me-3">
                                                <i class="bi bi-building"></i>
                                            </div>
                                            <div>
                                                <strong>{{ $office->name }}</strong>
                                                <br><small class="text-muted">Office ID: {{ $office->id }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            <i class="bi bi-geo-alt text-primary"></i> 
                                            <strong>{{ $office->district->name }}</strong>
                                            <br><span class="text-muted">{{ $office->district->province }}</span>
                                            <br><small>{{ Str::limit($office->address, 40) }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            <i class="bi bi-telephone text-success"></i> 
                                            <a href="tel:{{ $office->phone }}" class="text-decoration-none">{{ $office->phone }}</a>
                                            @if($office->email)
                                                <br><i class="bi bi-envelope text-info"></i> 
                                                <a href="mailto:{{ $office->email }}" class="text-decoration-none">{{ $office->email }}</a>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="text-center">
                                            <div class="avatar-circle bg-success text-white mb-2">
                                                {{ strtoupper(substr($office->manager_name, 0, 2)) }}
                                            </div>
                                            <strong class="small">{{ $office->manager_name }}</strong>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="text-center">
                                            @if($office->is_active)
                                                <span class="badge bg-success">Active</span>
                                                <br><small class="text-muted">Operational</small>
                                            @else
                                                <span class="badge bg-danger">Inactive</span>
                                                <br><small class="text-muted">Closed</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            {{ $office->created_at->format('M d, Y') }}
                                            <br>{{ $office->created_at->format('h:i A') }}
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('admin.offices.edit', $office) }}" 
                                               class="btn btn-sm btn-outline-primary" title="Edit Office">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-outline-info" 
                                                    data-bs-toggle="modal" data-bs-target="#viewModal{{ $office->id }}" 
                                                    title="View Details">
                                                <i class="bi bi-eye"></i>
                                            </button>
                                            <button type="button" class="btn btn-sm btn-outline-{{ $office->is_active ? 'warning' : 'success' }}" 
                                                    onclick="toggleOfficeStatus({{ $office->id }}, {{ $office->is_active ? 'false' : 'true' }})" 
                                                    title="{{ $office->is_active ? 'Deactivate' : 'Activate' }} Office">
                                                <i class="bi bi-{{ $office->is_active ? 'pause' : 'play' }}"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                
                                <!-- View Details Modal -->
                                <div class="modal fade" id="viewModal{{ $office->id }}" tabindex="-1">
                                    <div class="modal-dialog modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">Office Details - {{ $office->name }}</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                            </div>
                                            <div class="modal-body">
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <h6 class="text-primary mb-3">Office Information</h6>
                                                        <div class="mb-2"><strong>Name:</strong> {{ $office->name }}</div>
                                                        <div class="mb-2"><strong>District:</strong> {{ $office->district->name }}, {{ $office->district->province }}</div>
                                                        <div class="mb-2"><strong>Address:</strong> {{ $office->address }}</div>
                                                        <div class="mb-2"><strong>Status:</strong> 
                                                            <span class="badge bg-{{ $office->is_active ? 'success' : 'danger' }}">
                                                                {{ $office->is_active ? 'Active' : 'Inactive' }}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <h6 class="text-success mb-3">Contact & Management</h6>
                                                        <div class="mb-2"><strong>Phone:</strong> {{ $office->phone }}</div>
                                                        @if($office->email)
                                                            <div class="mb-2"><strong>Email:</strong> {{ $office->email }}</div>
                                                        @endif
                                                        <div class="mb-2"><strong>Manager:</strong> {{ $office->manager_name }}</div>
                                                        <div class="mb-2"><strong>Created:</strong> {{ $office->created_at->format('M d, Y') }}</div>
                                                    </div>
                                                </div>
                                                
                                                <!-- Office Statistics -->
                                                <hr>
                                                <h6 class="text-info mb-3">Office Statistics</h6>
                                                <div class="row text-center">
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-primary mb-1">0</div>
                                                            <small class="text-muted">Assigned Orders</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-success mb-1">0</div>
                                                            <small class="text-muted">Active Agents</small>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-4">
                                                        <div class="border rounded p-3">
                                                            <div class="h4 text-info mb-1">{{ $office->district->name }}</div>
                                                            <small class="text-muted">Service Area</small>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                <a href="{{ route('admin.offices.edit', $office) }}" class="btn btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit Office
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-4">
                    {{ $offices->appends(request()->query())->links() }}
                </div>
            @else
                <div class="text-center py-5">
                    <i class="bi bi-building text-muted" style="font-size: 4rem;"></i>
                    <h4 class="text-muted mt-3">No Offices Found</h4>
                    <p class="text-muted">No offices match your current search criteria.</p>
                    <a href="{{ route('admin.offices.create') }}" class="btn btn-primary">
                        <i class="bi bi-plus-circle"></i> Add First Office
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    .office-icon {
        width: 40px;
        height: 40px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 18px;
    }
    
    .avatar-circle {
        width: 35px;
        height: 35px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        font-size: 12px;
        margin: 0 auto;
    }
    
    .table th {
        border-top: none;
        font-weight: 600;
        font-size: 0.85rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table td {
        vertical-align: middle;
        font-size: 0.9rem;
    }
    
    .btn-group .btn {
        border-radius: 0.25rem;
        margin-right: 2px;
    }
    
    .badge {
        font-size: 0.75rem;
    }
    
    .card {
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        border: 1px solid #e3e6f0;
    }
    
    .card-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #e3e6f0;
    }
    
    .modal-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #dee2e6;
    }
</style>
@endpush

@push('scripts')
<script>
    function toggleOfficeStatus(officeId, newStatus) {
        if (confirm('Are you sure you want to ' + (newStatus ? 'activate' : 'deactivate') + ' this office?')) {
            // Create a form and submit it
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/offices/' + officeId + '/toggle-status';
            
            // Add CSRF token
            const csrfToken = document.createElement('input');
            csrfToken.type = 'hidden';
            csrfToken.name = '_token';
            csrfToken.value = '{{ csrf_token() }}';
            form.appendChild(csrfToken);
            
            // Add method override
            const methodField = document.createElement('input');
            methodField.type = 'hidden';
            methodField.name = '_method';
            methodField.value = 'PATCH';
            form.appendChild(methodField);
            
            // Add status field
            const statusField = document.createElement('input');
            statusField.type = 'hidden';
            statusField.name = 'is_active';
            statusField.value = newStatus;
            form.appendChild(statusField);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Show success message if exists
    @if(session('success'))
        const toast = document.createElement('div');
        toast.className = 'alert alert-success alert-dismissible fade show position-fixed';
        toast.style.top = '20px';
        toast.style.right = '20px';
        toast.style.zIndex = '9999';
        toast.innerHTML = `
            <i class="bi bi-check-circle"></i> {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.remove();
        }, 5000);
    @endif
</script>
@endpush