@extends('layouts.app')

@section('title', 'Edit District Pricing')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-pencil text-primary"></i> Edit District Pricing
            </h1>
            <p class="text-muted mb-0">Update delivery rates for {{ $pricing->pickupDistrict->name }} → {{ $pricing->dropDistrict->name }}</p>
        </div>
        <div>
            <a href="{{ route('admin.pricing') }}" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Pricing List
            </a>
        </div>
    </div>
    
    <!-- Pricing Form -->
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="bi bi-gear"></i> Pricing Configuration
                    </h6>
                </div>
                <div class="card-body">
                    <form method="POST" action="{{ route('admin.pricing.update', $pricing) }}" id="pricing-form">
                        @csrf
                        @method('PUT')
                        
                        <!-- Current Route Info -->
                        <div class="alert alert-info mb-4">
                            <div class="d-flex align-items-center">
                                <i class="bi bi-info-circle me-2"></i>
                                <div>
                                    <strong>Current Route:</strong> 
                                    {{ $pricing->pickupDistrict->name }} ({{ $pricing->pickupDistrict->province }}) 
                                    → 
                                    {{ $pricing->dropDistrict->name }} ({{ $pricing->dropDistrict->province }})
                                    <br>
                                    <strong>Service Type:</strong> {{ ucfirst($pricing->service_type) }}
                                </div>
                            </div>
                        </div>
                        
                        <!-- Route Configuration -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h5 class="text-primary mb-3">
                                    <i class="bi bi-geo-alt"></i> Route Configuration
                                </h5>
                            </div>
                            <div class="col-md-6">
                                <label for="pickup_district_id" class="form-label">Pickup District <span class="text-danger">*</span></label>
                                <select class="form-select @error('pickup_district_id') is-invalid @enderror" 
                                        id="pickup_district_id" name="pickup_district_id" required>
                                    <option value="">Select Pickup District</option>
                                    @foreach($districts as $district)
                                        <option value="{{ $district->id }}" 
                                                {{ (old('pickup_district_id', $pricing->pickup_district_id) == $district->id) ? 'selected' : '' }}>
                                            {{ $district->name }} ({{ $district->province }})
                                        </option>
                                    @endforeach
                                </select>
                                @error('pickup_district_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="drop_district_id" class="form-label">Drop District <span class="text-danger">*</span></label>
                                <select class="form-select @error('drop_district_id') is-invalid @enderror" 
                                        id="drop_district_id" name="drop_district_id" required>
                                    <option value="">Select Drop District</option>
                                    @foreach($districts as $district)
                                        <option value="{{ $district->id }}" 
                                                {{ (old('drop_district_id', $pricing->drop_district_id) == $district->id) ? 'selected' : '' }}>
                                            {{ $district->name }} ({{ $district->province }})
                                        </option>
                                    @endforeach
                                </select>
                                @error('drop_district_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Service Type -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h5 class="text-primary mb-3">
                                    <i class="bi bi-truck"></i> Service Configuration
                                </h5>
                            </div>
                            <div class="col-md-6">
                                <label for="service_type" class="form-label">Service Type <span class="text-danger">*</span></label>
                                <select class="form-select @error('service_type') is-invalid @enderror" 
                                        id="service_type" name="service_type" required>
                                    <option value="">Select Service Type</option>
                                    @foreach($serviceTypes as $key => $label)
                                        <option value="{{ $key }}" 
                                                {{ (old('service_type', $pricing->service_type) == $key) ? 'selected' : '' }}>
                                            {{ $label }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('service_type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="estimated_days" class="form-label">Estimated Delivery Days <span class="text-danger">*</span></label>
                                <input type="number" class="form-control @error('estimated_days') is-invalid @enderror" 
                                       id="estimated_days" name="estimated_days" min="1" max="30" 
                                       value="{{ old('estimated_days', $pricing->estimated_days) }}" required>
                                @error('estimated_days')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Pricing Details -->
                        <div class="row mb-4">
                            <div class="col-12">
                                <h5 class="text-primary mb-3">
                                    <i class="bi bi-currency-dollar"></i> Pricing Details
                                </h5>
                            </div>
                            <div class="col-md-6">
                                <label for="base_rate" class="form-label">Base Rate (Rs.)</label>
                                <input type="number" class="form-control @error('base_rate') is-invalid @enderror" 
                                       id="base_rate" name="base_rate" step="0.01" min="0" 
                                       value="{{ old('base_rate', $pricing->base_rate) }}" disabled>
                                <div class="form-text">Fixed charge regardless of weight (Disabled)</div>
                                @error('base_rate')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="per_kg_rate" class="form-label">Per KG Rate (Rs.) <span class="text-danger">*</span></label>
                                <input type="number" class="form-control @error('per_kg_rate') is-invalid @enderror" 
                                       id="per_kg_rate" name="per_kg_rate" step="0.01" min="0" 
                                       value="{{ old('per_kg_rate', $pricing->per_kg_rate) }}" required>
                                <div class="form-text">Additional charge per kilogram</div>
                                @error('per_kg_rate')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Min/Max Charges -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <label for="minimum_charge" class="form-label">Minimum Charge (Rs.)</label>
                                <input type="number" class="form-control @error('minimum_charge') is-invalid @enderror" 
                                       id="minimum_charge" name="minimum_charge" step="0.01" min="0" 
                                       value="{{ old('minimum_charge', $pricing->minimum_charge) }}" disabled>
                                <div class="form-text">Minimum amount to charge for this route (Disabled)</div>
                                @error('minimum_charge')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="maximum_charge" class="form-label">Maximum Charge (Rs.)</label>
                                <input type="number" class="form-control @error('maximum_charge') is-invalid @enderror" 
                                       id="maximum_charge" name="maximum_charge" step="0.01" min="0" 
                                       value="{{ old('maximum_charge', $pricing->maximum_charge) }}" disabled>
                                <div class="form-text">Optional maximum charge cap (Disabled)</div>
                                @error('maximum_charge')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Status and Notes -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                           value="1" {{ old('is_active', $pricing->is_active) ? 'checked' : '' }}>
                                    <label class="form-check-label" for="is_active">
                                        <strong>Active Status</strong>
                                        <div class="form-text">Enable this pricing configuration</div>
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <div class="row mb-4">
                            <div class="col-12">
                                <label for="notes" class="form-label">Notes</label>
                                <textarea class="form-control @error('notes') is-invalid @enderror" 
                                          id="notes" name="notes" rows="3" 
                                          placeholder="Optional notes about this pricing configuration">{{ old('notes', $pricing->notes) }}</textarea>
                                @error('notes')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Submit Buttons -->
                        <div class="row">
                            <div class="col-12">
                                <hr>
                                <div class="d-flex justify-content-end gap-2">
                                    <a href="{{ route('admin.pricing') }}" class="btn btn-secondary">
                                        <i class="bi bi-x-circle"></i> Cancel
                                    </a>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-check-circle"></i> Update Pricing
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Pricing Calculator Preview -->
        <div class="col-lg-4">
            <!-- Current Pricing Info -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-success">
                        <i class="bi bi-info-circle"></i> Current Pricing
                    </h6>
                </div>
                <div class="card-body">
                    <div class="current-pricing">
                        <div class="route-display mb-3">
                            <div class="d-flex align-items-center mb-2">
                                <i class="bi bi-geo-alt text-success me-2"></i>
                                <strong>{{ $pricing->pickupDistrict->name }}</strong>
                            </div>
                            <div class="d-flex align-items-center mb-2">
                                <i class="bi bi-arrow-down text-muted me-2"></i>
                            </div>
                            <div class="d-flex align-items-center">
                                <i class="bi bi-geo-alt-fill text-danger me-2"></i>
                                <strong>{{ $pricing->dropDistrict->name }}</strong>
                            </div>
                        </div>
                        
                        <div class="service-info mb-3">
                            <span class="badge bg-primary fs-6">{{ ucfirst($pricing->service_type) }}</span>
                            <span class="badge bg-info fs-6 ms-2">{{ $pricing->estimated_days }} days</span>
                            <span class="badge {{ $pricing->is_active ? 'bg-success' : 'bg-secondary' }} fs-6 ms-2">
                                {{ $pricing->is_active ? 'Active' : 'Inactive' }}
                            </span>
                        </div>
                        
                        <div class="pricing-breakdown">
                            <h6 class="text-success mb-2">Current Rates</h6>
                            <div><strong>Base Rate:</strong> Rs. {{ number_format($pricing->base_rate, 2) }}</div>
                            <div><strong>Per KG:</strong> Rs. {{ number_format($pricing->per_kg_rate, 2) }}</div>
                            <div><strong>Minimum:</strong> Rs. {{ number_format($pricing->minimum_charge, 2) }}</div>
                            @if($pricing->maximum_charge)
                                <div><strong>Maximum:</strong> Rs. {{ number_format($pricing->maximum_charge, 2) }}</div>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Pricing Calculator Preview -->
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="bi bi-calculator"></i> Updated Preview
                    </h6>
                </div>
                <div class="card-body">
                    <div id="pricing-preview">
                        <!-- Will be populated by JavaScript -->
                    </div>
                    
                    <!-- Sample Calculation -->
                    <div class="mt-4">
                        <h6 class="text-primary">Test Calculation</h6>
                        <div class="input-group mb-3">
                            <span class="input-group-text">Weight (KG)</span>
                            <input type="number" class="form-control" id="test-weight" 
                                   placeholder="Enter weight" min="0.1" step="0.1" value="1">
                            <button class="btn btn-outline-primary" type="button" id="calculate-btn">
                                Calculate
                            </button>
                        </div>
                        <div id="calculation-result" class="alert alert-info d-none">
                            <strong>Total Cost: Rs. <span id="total-cost">0.00</span></strong>
                            <div class="mt-2 small">
                                <div>Base Rate: Rs. <span id="calc-base">0.00</span></div>
                                <div>Weight Charge: Rs. <span id="calc-weight">0.00</span></div>
                                <div>Final Amount: Rs. <span id="calc-final">0.00</span></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Change History -->
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-warning">
                        <i class="bi bi-clock-history"></i> Pricing History
                    </h6>
                </div>
                <div class="card-body">
                    <div class="timeline">
                        <div class="timeline-item">
                            <div class="timeline-marker bg-primary"></div>
                            <div class="timeline-content">
                                <h6 class="mb-1">Created</h6>
                                <p class="text-muted mb-0 small">{{ $pricing->created_at->format('M d, Y H:i') }}</p>
                            </div>
                        </div>
                        @if($pricing->updated_at != $pricing->created_at)
                            <div class="timeline-item">
                                <div class="timeline-marker bg-warning"></div>
                                <div class="timeline-content">
                                    <h6 class="mb-1">Last Updated</h6>
                                    <p class="text-muted mb-0 small">{{ $pricing->updated_at->format('M d, Y H:i') }}</p>
                                </div>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('pricing-form');
    const previewDiv = document.getElementById('pricing-preview');
    const calculateBtn = document.getElementById('calculate-btn');
    const testWeightInput = document.getElementById('test-weight');
    const resultDiv = document.getElementById('calculation-result');
    
    // Initialize preview
    updatePreview();
    
    // Update preview when form values change
    function updatePreview() {
        const pickupDistrict = document.getElementById('pickup_district_id');
        const dropDistrict = document.getElementById('drop_district_id');
        const serviceType = document.getElementById('service_type');
        const baseRate = document.getElementById('base_rate').value;
        const perKgRate = document.getElementById('per_kg_rate').value;
        const minCharge = document.getElementById('minimum_charge').value;
        const maxCharge = document.getElementById('maximum_charge').value;
        const estimatedDays = document.getElementById('estimated_days').value;
        
        if (pickupDistrict.value && dropDistrict.value && serviceType.value) {
            const pickupText = pickupDistrict.options[pickupDistrict.selectedIndex].text;
            const dropText = dropDistrict.options[dropDistrict.selectedIndex].text;
            const serviceText = serviceType.options[serviceType.selectedIndex].text;
            
            previewDiv.innerHTML = `
                <div class="pricing-summary">
                    <h6 class="text-primary mb-3">Updated Route</h6>
                    <div class="route-display mb-3">
                        <div class="d-flex align-items-center mb-2">
                            <i class="bi bi-geo-alt text-success me-2"></i>
                            <strong>${pickupText}</strong>
                        </div>
                        <div class="d-flex align-items-center mb-2">
                            <i class="bi bi-arrow-down text-muted me-2"></i>
                        </div>
                        <div class="d-flex align-items-center">
                            <i class="bi bi-geo-alt-fill text-danger me-2"></i>
                            <strong>${dropText}</strong>
                        </div>
                    </div>
                    
                    <div class="service-info mb-3">
                        <span class="badge bg-primary fs-6">${serviceText}</span>
                        ${estimatedDays ? `<span class="badge bg-info fs-6 ms-2">${estimatedDays} days</span>` : ''}
                    </div>
                    
                    <div class="pricing-breakdown">
                        <h6 class="text-primary mb-2">Updated Rates</h6>
                        ${perKgRate ? `<div><strong>Per KG Rate:</strong> Rs. ${parseFloat(perKgRate).toFixed(2)}</div>` : ''}
                        <div class="text-muted small">Base Rate, Minimum & Maximum charges are disabled</div>
                    </div>
                </div>
            `;
        }
    }
    
    // Calculate pricing for test weight
    function calculateTestPrice() {
        const weight = parseFloat(testWeightInput.value) || 0;
        const perKgRate = parseFloat(document.getElementById('per_kg_rate').value) || 0;
        
        if (weight > 0 && perKgRate > 0) {
            const finalAmount = weight * perKgRate;
            
            document.getElementById('calc-base').textContent = '0.00';
            document.getElementById('calc-weight').textContent = finalAmount.toFixed(2);
            document.getElementById('calc-final').textContent = finalAmount.toFixed(2);
            document.getElementById('total-cost').textContent = finalAmount.toFixed(2);
            
            resultDiv.classList.remove('d-none');
        } else {
            resultDiv.classList.add('d-none');
        }
    }
    
    // Event listeners
    form.addEventListener('input', updatePreview);
    form.addEventListener('change', updatePreview);
    calculateBtn.addEventListener('click', calculateTestPrice);
    testWeightInput.addEventListener('input', calculateTestPrice);
    
    // Update preview when districts change
    document.getElementById('pickup_district_id').addEventListener('change', updatePreview);
    document.getElementById('drop_district_id').addEventListener('change', updatePreview);
    
    // Minimum charge validation
    document.getElementById('minimum_charge').addEventListener('input', function() {
        const maxChargeInput = document.getElementById('maximum_charge');
        const minValue = parseFloat(this.value);
        const maxValue = parseFloat(maxChargeInput.value);
        
        if (maxValue > 0 && minValue > maxValue) {
            maxChargeInput.setCustomValidity('Maximum charge must be greater than minimum charge');
        } else {
            maxChargeInput.setCustomValidity('');
        }
    });
    
    document.getElementById('maximum_charge').addEventListener('input', function() {
        const minChargeInput = document.getElementById('minimum_charge');
        const maxValue = parseFloat(this.value);
        const minValue = parseFloat(minChargeInput.value);
        
        if (maxValue > 0 && minValue > maxValue) {
            this.setCustomValidity('Maximum charge must be greater than minimum charge');
        } else {
            this.setCustomValidity('');
        }
    });
});
</script>

<style>
.route-display {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    border-left: 4px solid #007bff;
}

.pricing-breakdown div {
    padding: 2px 0;
    border-bottom: 1px solid #eee;
}

.pricing-breakdown div:last-child {
    border-bottom: none;
}

.service-info .badge {
    font-size: 0.8rem;
}

.timeline {
    position: relative;
    padding-left: 30px;
}

.timeline-item {
    position: relative;
    margin-bottom: 20px;
}

.timeline-marker {
    position: absolute;
    left: -35px;
    top: 5px;
    width: 12px;
    height: 12px;
    border-radius: 50%;
}

.timeline-item:not(:last-child)::before {
    content: '';
    position: absolute;
    left: -30px;
    top: 17px;
    width: 2px;
    height: calc(100% + 5px);
    background: #dee2e6;
}

.current-pricing .route-display {
    border-left-color: #28a745;
}
</style>
@endsection