@extends('layouts.app')

@section('title', 'District Pricing Management')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-currency-dollar text-primary"></i> District Pricing Management
            </h1>
            <p class="text-muted mb-0">Manage delivery rates based on geographic locations</p>
        </div>
        <div>
            <a href="{{ route('admin.dashboard') }}" class="btn btn-outline-secondary me-2">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
            <a href="{{ route('admin.pricing.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Add New Pricing
            </a>
        </div>
    </div>
    
    <!-- Search & Filters -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-funnel"></i> Search & Filters
            </h6>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('admin.pricing') }}">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label for="pickup_district" class="form-label">Pickup District</label>
                        <select class="form-select" id="pickup_district" name="pickup_district">
                            <option value="">All Districts</option>
                            @foreach($districts as $district)
                                <option value="{{ $district->id }}" {{ request('pickup_district') == $district->id ? 'selected' : '' }}>
                                    {{ $district->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="drop_district" class="form-label">Drop District</label>
                        <select class="form-select" id="drop_district" name="drop_district">
                            <option value="">All Districts</option>
                            @foreach($districts as $district)
                                <option value="{{ $district->id }}" {{ request('drop_district') == $district->id ? 'selected' : '' }}>
                                    {{ $district->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="service_type" class="form-label">Service Type</label>
                        <select class="form-select" id="service_type" name="service_type">
                            <option value="">All Services</option>
                            @foreach($serviceTypes as $key => $label)
                                <option value="{{ $key }}" {{ request('service_type') == $key ? 'selected' : '' }}>
                                    {{ $label }}
                                </option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="bi bi-search"></i> Search
                        </button>
                        <a href="{{ route('admin.pricing') }}" class="btn btn-outline-secondary">
                            <i class="bi bi-x-circle"></i> Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Pricing Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-list"></i> Pricing List ({{ $pricings->total() }} total)
            </h6>
            <div class="dropdown">
                <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="bi bi-download"></i> Export
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-excel"></i> Export to Excel</a></li>
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-pdf"></i> Export to PDF</a></li>
                </ul>
            </div>
        </div>
        <div class="card-body">
            @if($pricings->count() > 0)
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th>Route</th>
                                <th>Service Type</th>
                                <th>Pricing Details</th>
                                <th>Delivery Time</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($pricings as $pricing)
                                <tr>
                                    <td>
                                        <div class="route-info">
                                            <div class="d-flex align-items-center mb-1">
                                                <i class="bi bi-geo-alt text-success me-2"></i>
                                                <strong>{{ $pricing->pickupDistrict->name }}</strong>
                                            </div>
                                            <div class="d-flex align-items-center">
                                                <i class="bi bi-geo-alt-fill text-danger me-2"></i>
                                                <strong>{{ $pricing->dropDistrict->name }}</strong>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge 
                                            @if($pricing->service_type === 'standard') bg-primary
                                            @elseif($pricing->service_type === 'express') bg-warning
                                            @else bg-danger
                                            @endif fs-6">
                                            {{ ucfirst($pricing->service_type) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="pricing-details">
                                            <div><strong>Base:</strong> Rs. {{ number_format($pricing->base_rate, 2) }}</div>
                                            <div><strong>Per KG:</strong> Rs. {{ number_format($pricing->per_kg_rate, 2) }}</div>
                                            <div><strong>Min:</strong> Rs. {{ number_format($pricing->minimum_charge, 2) }}</div>
                                            @if($pricing->maximum_charge)
                                                <div><strong>Max:</strong> Rs. {{ number_format($pricing->maximum_charge, 2) }}</div>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge bg-info fs-6">
                                            {{ $pricing->estimated_days }} {{ $pricing->estimated_days == 1 ? 'day' : 'days' }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="form-check form-switch">
                                            <input class="form-check-input status-toggle" type="checkbox" 
                                                   data-pricing-id="{{ $pricing->id }}"
                                                   {{ $pricing->is_active ? 'checked' : '' }}>
                                            <label class="form-check-label">
                                                <span class="badge {{ $pricing->is_active ? 'bg-success' : 'bg-secondary' }}">
                                                    {{ $pricing->is_active ? 'Active' : 'Inactive' }}
                                                </span>
                                            </label>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('admin.pricing.edit', $pricing) }}" 
                                               class="btn btn-sm btn-outline-primary" title="Edit">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-outline-danger delete-pricing" 
                                                    data-pricing-id="{{ $pricing->id }}"
                                                    data-route="{{ $pricing->pickupDistrict->name }} → {{ $pricing->dropDistrict->name }}"
                                                    data-service="{{ ucfirst($pricing->service_type) }}"
                                                    title="Delete">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-4">
                    <div class="text-muted">
                        Showing {{ $pricings->firstItem() }} to {{ $pricings->lastItem() }} of {{ $pricings->total() }} results
                    </div>
                    {{ $pricings->links() }}
                </div>
            @else
                <div class="text-center py-5">
                    <i class="bi bi-currency-dollar text-muted" style="font-size: 3rem;"></i>
                    <h5 class="text-muted mt-3">No pricing configurations found</h5>
                    <p class="text-muted">Start by creating your first district pricing configuration.</p>
                    <a href="{{ route('admin.pricing.create') }}" class="btn btn-primary">
                        <i class="bi bi-plus-circle"></i> Add New Pricing
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deletePricingModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Deletion</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this pricing configuration?</p>
                <div class="alert alert-warning">
                    <strong>Route:</strong> <span id="delete-route"></span><br>
                    <strong>Service:</strong> <span id="delete-service"></span>
                </div>
                <p class="text-danger"><strong>This action cannot be undone.</strong></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="delete-form" method="POST" style="display: inline;">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete Pricing</button>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
.district-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
}

.route-info {
    min-width: 200px;
}

.pricing-details {
    font-size: 0.9rem;
}

.pricing-details div {
    margin-bottom: 2px;
}

.status-toggle {
    cursor: pointer;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Status toggle functionality
    document.querySelectorAll('.status-toggle').forEach(function(toggle) {
        toggle.addEventListener('change', function() {
            const pricingId = this.dataset.pricingId;
            const isActive = this.checked;
            
            fetch(`/admin/pricing/${pricingId}/toggle-status`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    is_active: isActive
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update badge
                    const badge = this.parentElement.querySelector('.badge');
                    if (isActive) {
                        badge.className = 'badge bg-success';
                        badge.textContent = 'Active';
                    } else {
                        badge.className = 'badge bg-secondary';
                        badge.textContent = 'Inactive';
                    }
                    
                    // Show success message
                    showAlert('success', 'Pricing status updated successfully!');
                } else {
                    // Revert toggle
                    this.checked = !isActive;
                    showAlert('error', 'Failed to update pricing status.');
                }
            })
            .catch(error => {
                // Revert toggle
                this.checked = !isActive;
                showAlert('error', 'An error occurred while updating pricing status.');
            });
        });
    });
    
    // Delete pricing functionality
    document.querySelectorAll('.delete-pricing').forEach(function(button) {
        button.addEventListener('click', function() {
            const pricingId = this.dataset.pricingId;
            const route = this.dataset.route;
            const service = this.dataset.service;
            
            document.getElementById('delete-route').textContent = route;
            document.getElementById('delete-service').textContent = service;
            document.getElementById('delete-form').action = `/admin/pricing/${pricingId}`;
            
            new bootstrap.Modal(document.getElementById('deletePricingModal')).show();
        });
    });
});

function showAlert(type, message) {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type === 'success' ? 'success' : 'danger'} alert-dismissible fade show`;
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    const container = document.querySelector('.container-fluid');
    container.insertBefore(alertDiv, container.firstChild);
    
    // Auto dismiss after 5 seconds
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}
</script>
@endsection