@extends('layouts.app')

@section('title', 'Manage Users')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-0 text-gray-800">
                <i class="bi bi-people text-primary"></i> User Management
            </h1>
            <p class="text-muted mb-0">Manage users and assign roles</p>
        </div>
        <div>
            <a href="{{ route('admin.dashboard') }}" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
        </div>
    </div>
    
    <!-- Search & Filters -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-funnel"></i> Search & Filters
            </h6>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('admin.users') }}">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label for="search" class="form-label">Search</label>
                        <input type="text" class="form-control" id="search" name="search" 
                               value="{{ request('search') }}" 
                               placeholder="Name, email, phone...">
                    </div>
                    <div class="col-md-3">
                        <label for="role" class="form-label">Role</label>
                        <select class="form-select" id="role" name="role">
                            <option value="">All Roles</option>
                            <option value="customer" {{ request('role') == 'customer' ? 'selected' : '' }}>Customer</option>
                            <option value="agent" {{ request('role') == 'agent' ? 'selected' : '' }}>Agent</option>
                            <option value="admin" {{ request('role') == 'admin' ? 'selected' : '' }}>Admin</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="status" class="form-label">Status</label>
                        <select class="form-select" id="status" name="status">
                            <option value="">All Status</option>
                            <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                            <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                        </select>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary me-2">
                            <i class="bi bi-search"></i> Filter
                        </button>
                        <a href="{{ route('admin.users') }}" class="btn btn-outline-secondary">
                            <i class="bi bi-x-circle"></i> Clear
                        </a>
                    </div>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Users Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex flex-row align-items-center justify-content-between">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="bi bi-list"></i> Users List ({{ $users->total() }} total)
            </h6>
            <div class="dropdown">
                <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="bi bi-download"></i> Export
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-excel"></i> Export to Excel</a></li>
                    <li><a class="dropdown-item" href="#"><i class="bi bi-file-earmark-pdf"></i> Export to PDF</a></li>
                </ul>
            </div>
        </div>
        <div class="card-body">
            @if($users->count() > 0)
                <div class="table-responsive">
                    <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                        <thead class="table-light">
                            <tr>
                                <th>User Info</th>
                                <th>Contact</th>
                                <th>Role</th>
                                <th>Activity</th>
                                <th>Status</th>
                                <th>Joined Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($users as $user)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="user-avatar bg-{{ $user->role === 'admin' ? 'danger' : ($user->role === 'agent' ? 'warning' : 'primary') }} text-white me-3">
                                                {{ strtoupper(substr($user->name, 0, 2)) }}
                                            </div>
                                            <div>
                                                <strong>{{ $user->name }}</strong>
                                                <br><small class="text-muted">ID: {{ $user->id }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            <i class="bi bi-envelope text-primary"></i> 
                                            <a href="mailto:{{ $user->email }}" class="text-decoration-none">{{ $user->email }}</a>
                                            @if($user->phone)
                                                <br><i class="bi bi-telephone text-success"></i> 
                                                <a href="tel:{{ $user->phone }}" class="text-decoration-none">{{ $user->phone }}</a>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge bg-{{ $user->role === 'admin' ? 'danger' : ($user->role === 'agent' ? 'warning' : 'primary') }} fs-6">
                                            {{ ucfirst($user->role) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="text-center">
                                            <div class="h6 text-info mb-1">{{ $user->orders_count }}</div>
                                            <small class="text-muted">Orders</small>
                                            @if($user->agent)
                                                <br><div class="h6 text-success mb-1">{{ $user->agent->assignedOrders()->count() }}</div>
                                                <small class="text-muted">Assigned</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="text-center">
                                            @if($user->is_active)
                                                <span class="badge bg-success">Active</span>
                                                <br><small class="text-muted">Online</small>
                                            @else
                                                <span class="badge bg-danger">Inactive</span>
                                                <br><small class="text-muted">Offline</small>
                                            @endif
                                        </div>
                                    </td>
                                    <td>
                                        <div class="small">
                                            {{ $user->created_at->format('M d, Y') }}
                                            <br>{{ $user->created_at->format('h:i A') }}
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('admin.users.edit', $user) }}" 
                                               class="btn btn-sm btn-outline-primary" title="Edit User">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <button type="button" class="btn btn-sm btn-outline-info" 
                                                    data-bs-toggle="modal" data-bs-target="#viewModal{{ $user->id }}" 
                                                    title="View Details">
                                                <i class="bi bi-eye"></i>
                                            </button>
                                            <button type="button" class="btn btn-sm btn-outline-{{ $user->is_active ? 'warning' : 'success' }}" 
                                                    onclick="toggleUserStatus({{ $user->id }}, {{ $user->is_active ? 'false' : 'true' }})" 
                                                    title="{{ $user->is_active ? 'Deactivate' : 'Activate' }} User">
                                                <i class="bi bi-{{ $user->is_active ? 'pause' : 'play' }}"></i>
                                            </button>
                                            @if($user->role !== 'admin' && $user->orders_count == 0)
                                                <button type="button" class="btn btn-sm btn-outline-danger" 
                                                        onclick="deleteUser({{ $user->id }})" 
                                                        title="Delete User">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                
                                <!-- View Details Modal -->
                                <div class="modal fade" id="viewModal{{ $user->id }}" tabindex="-1">
                                    <div class="modal-dialog modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">User Details - {{ $user->name }}</h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                            </div>
                                            <div class="modal-body">
                                                <div class="row">
                                                    <div class="col-md-6">
                                                        <h6 class="text-primary mb-3">Personal Information</h6>
                                                        <div class="mb-2"><strong>Name:</strong> {{ $user->name }}</div>
                                                        <div class="mb-2"><strong>Email:</strong> {{ $user->email }}</div>
                                                        <div class="mb-2"><strong>Phone:</strong> {{ $user->phone ?? 'Not provided' }}</div>
                                                        <div class="mb-2"><strong>Role:</strong> 
                                                            <span class="badge bg-{{ $user->role === 'admin' ? 'danger' : ($user->role === 'agent' ? 'warning' : 'primary') }}">
                                                                {{ ucfirst($user->role) }}
                                                            </span>
                                                        </div>
                                                        <div class="mb-2"><strong>Status:</strong> 
                                                            <span class="badge bg-{{ $user->is_active ? 'success' : 'danger' }}">
                                                                {{ $user->is_active ? 'Active' : 'Inactive' }}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <h6 class="text-success mb-3">Account Activity</h6>
                                                        <div class="mb-2"><strong>Total Orders:</strong> {{ $user->orders_count }}</div>
                                                        @if($user->agent)
                                                            <div class="mb-2"><strong>Agent Code:</strong> {{ $user->agent->agent_code }}</div>
                                                            <div class="mb-2"><strong>Assigned Orders:</strong> {{ $user->agent->assignedOrders()->count() }}</div>
                                                            <div class="mb-2"><strong>Vehicle Type:</strong> {{ ucfirst($user->agent->vehicle_type) }}</div>
                                                        @endif
                                                        <div class="mb-2"><strong>Joined:</strong> {{ $user->created_at->format('M d, Y') }}</div>
                                                        <div class="mb-2"><strong>Last Updated:</strong> {{ $user->updated_at->format('M d, Y h:i A') }}</div>
                                                    </div>
                                                </div>
                                                
                                                @if($user->orders_count > 0)
                                                    <hr>
                                                    <h6 class="text-info mb-3">Recent Orders</h6>
                                                    <div class="table-responsive">
                                                        <table class="table table-sm">
                                                            <thead>
                                                                <tr>
                                                                    <th>Tracking Code</th>
                                                                    <th>Route</th>
                                                                    <th>Status</th>
                                                                    <th>Date</th>
                                                                </tr>
                                                            </thead>
                                                            <tbody>
                                                                @foreach($user->orders()->latest()->take(5)->get() as $order)
                                                                    <tr>
                                                                        <td><strong>{{ $order->tracking_code }}</strong></td>
                                                                        <td class="small">
                                                                            {{ $order->pickupDistrict->name }} → {{ $order->dropDistrict->name }}
                                                                        </td>
                                                                        <td>
                                                                            <span class="badge bg-{{ $order->status_badge }} badge-sm">
                                                                                {{ ucfirst(str_replace('_', ' ', $order->status)) }}
                                                                            </span>
                                                                        </td>
                                                                        <td class="small">{{ $order->created_at->format('M d') }}</td>
                                                                    </tr>
                                                                @endforeach
                                                            </tbody>
                                                        </table>
                                                    </div>
                                                @endif
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                <a href="{{ route('admin.users.edit', $user) }}" class="btn btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit User
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-4">
                    {{ $users->appends(request()->query())->links() }}
                </div>
            @else
                <div class="text-center py-5">
                    <i class="bi bi-people text-muted" style="font-size: 4rem;"></i>
                    <h4 class="text-muted mt-3">No Users Found</h4>
                    <p class="text-muted">No users match your current search criteria.</p>
                    <a href="{{ route('admin.users') }}" class="btn btn-primary">
                        <i class="bi bi-arrow-clockwise"></i> Reset Filters
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    .user-avatar {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        font-size: 14px;
    }
    
    .table th {
        border-top: none;
        font-weight: 600;
        font-size: 0.85rem;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table td {
        vertical-align: middle;
        font-size: 0.9rem;
    }
    
    .btn-group .btn {
        border-radius: 0.25rem;
        margin-right: 2px;
    }
    
    .badge {
        font-size: 0.75rem;
    }
    
    .badge-sm {
        font-size: 0.65rem;
        padding: 0.25em 0.5em;
    }
    
    .card {
        box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        border: 1px solid #e3e6f0;
    }
    
    .card-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #e3e6f0;
    }
    
    .modal-header {
        background-color: #f8f9fa;
        border-bottom: 1px solid #dee2e6;
    }
</style>
@endpush

@push('scripts')
<script>
    function toggleUserStatus(userId, newStatus) {
        if (confirm('Are you sure you want to ' + (newStatus ? 'activate' : 'deactivate') + ' this user?')) {
            // Create a form and submit it
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/users/' + userId + '/toggle-status';
            
            // Add CSRF token
            const csrfToken = document.createElement('input');
            csrfToken.type = 'hidden';
            csrfToken.name = '_token';
            csrfToken.value = '{{ csrf_token() }}';
            form.appendChild(csrfToken);
            
            // Add method override
            const methodField = document.createElement('input');
            methodField.type = 'hidden';
            methodField.name = '_method';
            methodField.value = 'PATCH';
            form.appendChild(methodField);
            
            // Add status field
            const statusField = document.createElement('input');
            statusField.type = 'hidden';
            statusField.name = 'is_active';
            statusField.value = newStatus;
            form.appendChild(statusField);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    function deleteUser(userId) {
        if (confirm('Are you sure you want to delete this user? This action cannot be undone.')) {
            // Create a form and submit it
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/users/' + userId;
            
            // Add CSRF token
            const csrfToken = document.createElement('input');
            csrfToken.type = 'hidden';
            csrfToken.name = '_token';
            csrfToken.value = '{{ csrf_token() }}';
            form.appendChild(csrfToken);
            
            // Add method override
            const methodField = document.createElement('input');
            methodField.type = 'hidden';
            methodField.name = '_method';
            methodField.value = 'DELETE';
            form.appendChild(methodField);
            
            document.body.appendChild(form);
            form.submit();
        }
    }
    
    // Show success/error messages
    @if(session('success'))
        const successToast = document.createElement('div');
        successToast.className = 'alert alert-success alert-dismissible fade show position-fixed';
        successToast.style.top = '20px';
        successToast.style.right = '20px';
        successToast.style.zIndex = '9999';
        successToast.innerHTML = `
            <i class="bi bi-check-circle"></i> {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        document.body.appendChild(successToast);
        
        setTimeout(() => {
            successToast.remove();
        }, 5000);
    @endif
    
    @if(session('error'))
        const errorToast = document.createElement('div');
        errorToast.className = 'alert alert-danger alert-dismissible fade show position-fixed';
        errorToast.style.top = '20px';
        errorToast.style.right = '20px';
        errorToast.style.zIndex = '9999';
        errorToast.innerHTML = `
            <i class="bi bi-exclamation-triangle"></i> {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        document.body.appendChild(errorToast);
        
        setTimeout(() => {
            errorToast.remove();
        }, 5000);
    @endif
</script>
@endpush