@extends('layouts.app')

@section('title', 'Book Courier')

@section('content')
<div class="container py-4">
    <div class="row">
        <div class="col-lg-8">
            <div class="card shadow">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0"><i class="bi bi-plus-circle"></i> Book Courier Service</h4>
                </div>
                <div class="card-body">
                    <form method="POST" action="{{ route('courier.store') }}" id="courierForm">
                        @csrf
                        
                        <!-- Pickup & Drop Districts -->
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <label for="pickup_district_id" class="form-label">Pickup District</label>
                                <select class="form-select @error('pickup_district_id') is-invalid @enderror" 
                                        id="pickup_district_id" name="pickup_district_id" required>
                                    <option value="">Select Pickup District</option>
                                    @foreach($districts as $district)
                                        <option value="{{ $district->id }}" {{ old('pickup_district_id') == $district->id ? 'selected' : '' }}>
                                            {{ $district->name }}, {{ $district->province }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('pickup_district_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="drop_district_id" class="form-label">Drop District</label>
                                <select class="form-select @error('drop_district_id') is-invalid @enderror" 
                                        id="drop_district_id" name="drop_district_id" required>
                                    <option value="">Select Drop District</option>
                                    @foreach($districts as $district)
                                        <option value="{{ $district->id }}" {{ old('drop_district_id') == $district->id ? 'selected' : '' }}>
                                            {{ $district->name }}, {{ $district->province }}
                                        </option>
                                    @endforeach
                                </select>
                                @error('drop_district_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Sender Information -->
                        <h5 class="mb-3"><i class="bi bi-person"></i> Sender Information</h5>
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <label for="sender_name" class="form-label">Sender Name</label>
                                <input type="text" class="form-control @error('sender_name') is-invalid @enderror" 
                                       id="sender_name" name="sender_name" value="{{ old('sender_name', auth()->user()->name) }}" required>
                                @error('sender_name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="sender_phone" class="form-label">Sender Phone</label>
                                <input type="tel" class="form-control @error('sender_phone') is-invalid @enderror" 
                                       id="sender_phone" name="sender_phone" value="{{ old('sender_phone', auth()->user()->phone) }}" required>
                                @error('sender_phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="mb-4">
                            <label for="sender_address" class="form-label">Sender Address</label>
                            <textarea class="form-control @error('sender_address') is-invalid @enderror" 
                                      id="sender_address" name="sender_address" rows="2" required>{{ old('sender_address') }}</textarea>
                            @error('sender_address')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <!-- Receiver Information -->
                        <h5 class="mb-3"><i class="bi bi-person-check"></i> Receiver Information</h5>
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <label for="receiver_name" class="form-label">Receiver Name</label>
                                <input type="text" class="form-control @error('receiver_name') is-invalid @enderror" 
                                       id="receiver_name" name="receiver_name" value="{{ old('receiver_name') }}" required>
                                @error('receiver_name')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="receiver_phone" class="form-label">Receiver Phone</label>
                                <input type="tel" class="form-control @error('receiver_phone') is-invalid @enderror" 
                                       id="receiver_phone" name="receiver_phone" value="{{ old('receiver_phone') }}" required>
                                @error('receiver_phone')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="mb-4">
                            <label for="receiver_address" class="form-label">Receiver Address</label>
                            <textarea class="form-control @error('receiver_address') is-invalid @enderror" 
                                      id="receiver_address" name="receiver_address" rows="2" required>{{ old('receiver_address') }}</textarea>
                            @error('receiver_address')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <!-- Parcel Details -->
                        <h5 class="mb-3"><i class="bi bi-box"></i> Parcel Details</h5>
                        <div class="row mb-4">
                            <div class="col-md-4">
                                <label for="weight" class="form-label">Weight (kg)</label>
                                <input type="number" class="form-control @error('weight') is-invalid @enderror" 
                                       id="weight" name="weight" value="{{ old('weight') }}" 
                                       step="0.1" min="0.1" required>
                                @error('weight')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label for="size" class="form-label">Size</label>
                                <select class="form-select @error('size') is-invalid @enderror" 
                                        id="size" name="size" required>
                                    <option value="">Select Size</option>
                                    <option value="Small" {{ old('size') == 'Small' ? 'selected' : '' }}>Small (up to 30cm)</option>
                                    <option value="Medium" {{ old('size') == 'Medium' ? 'selected' : '' }}>Medium (30-60cm)</option>
                                    <option value="Large" {{ old('size') == 'Large' ? 'selected' : '' }}>Large (60cm+)</option>
                                </select>
                                @error('size')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label for="category" class="form-label">Category</label>
                                <select class="form-select @error('category') is-invalid @enderror" 
                                        id="category" name="category" required>
                                    <option value="">Select Category</option>
                                    <option value="Documents" {{ old('category') == 'Documents' ? 'selected' : '' }}>Documents</option>
                                    <option value="Electronics" {{ old('category') == 'Electronics' ? 'selected' : '' }}>Electronics</option>
                                    <option value="Clothing" {{ old('category') == 'Clothing' ? 'selected' : '' }}>Clothing</option>
                                    <option value="Food" {{ old('category') == 'Food' ? 'selected' : '' }}>Food Items</option>
                                    <option value="Fragile" {{ old('category') == 'Fragile' ? 'selected' : '' }}>Fragile Items</option>
                                    <option value="Other" {{ old('category') == 'Other' ? 'selected' : '' }}>Other</option>
                                </select>
                                @error('category')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="mb-4">
                            <label for="description" class="form-label">Description (Optional)</label>
                            <textarea class="form-control" id="description" name="description" 
                                      rows="2" placeholder="Brief description of the package">{{ old('description') }}</textarea>
                        </div>
                        
                        <!-- Delivery Options -->
                        <h5 class="mb-3"><i class="bi bi-truck"></i> Delivery Options</h5>
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <label for="delivery_type" class="form-label">Delivery Type</label>
                                <select class="form-select @error('delivery_type') is-invalid @enderror" 
                                        id="delivery_type" name="delivery_type" required>
                                    <option value="">Select Delivery Type</option>
                                    <option value="standard" {{ old('delivery_type') == 'standard' ? 'selected' : '' }}>Standard (3-5 days)</option>
                                    <option value="express" {{ old('delivery_type') == 'express' ? 'selected' : '' }}>Express (1-2 days)</option>
                                    <option value="same_day" {{ old('delivery_type') == 'same_day' ? 'selected' : '' }}>Same Day</option>
                                </select>
                                @error('delivery_type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="vehicle_type" class="form-label">Vehicle Type</label>
                                <select class="form-select @error('vehicle_type') is-invalid @enderror" 
                                        id="vehicle_type" name="vehicle_type" required>
                                    <option value="">Select Vehicle</option>
                                    <option value="bike" {{ old('vehicle_type') == 'bike' ? 'selected' : '' }}>Bike</option>
                                    <option value="car" {{ old('vehicle_type') == 'car' ? 'selected' : '' }}>Car</option>
                                    <option value="van" {{ old('vehicle_type') == 'van' ? 'selected' : '' }}>Van</option>
                                    <option value="truck" {{ old('vehicle_type') == 'truck' ? 'selected' : '' }}>Truck</option>
                                </select>
                                @error('vehicle_type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        
                        <!-- Payment Method -->
                        <h5 class="mb-3"><i class="bi bi-credit-card"></i> Payment Method</h5>
                        <div class="mb-4">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="payment_method" 
                                               id="cod" value="cod" {{ old('payment_method', 'cod') == 'cod' ? 'checked' : '' }}>
                                        <label class="form-check-label" for="cod">
                                            <i class="bi bi-cash-coin"></i> Cash on Delivery
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="payment_method" 
                                               id="jazzcash" value="jazzcash" {{ old('payment_method') == 'jazzcash' ? 'checked' : '' }}>
                                        <label class="form-check-label" for="jazzcash">
                                            <i class="bi bi-phone"></i> JazzCash
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="radio" name="payment_method" 
                                               id="easypaisa" value="easypaisa" {{ old('payment_method') == 'easypaisa' ? 'checked' : '' }}>
                                        <label class="form-check-label" for="easypaisa">
                                            <i class="bi bi-wallet2"></i> Easypaisa
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Submit Button -->
                        <div class="d-grid">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="bi bi-check-circle"></i> Calculate Price & Book
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Price Estimation Sidebar -->
        <div class="col-lg-4">
            <div class="card shadow">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0"><i class="bi bi-calculator"></i> Price Estimation</h5>
                </div>
                <div class="card-body">
                    <div id="priceEstimation">
                        <p class="text-muted">Fill the form to get price estimation</p>
                    </div>
                </div>
            </div>
            
            <!-- Service Info -->
            <div class="card shadow mt-4">
                <div class="card-header">
                    <h6 class="mb-0"><i class="bi bi-info-circle"></i> Service Information</h6>
                </div>
                <div class="card-body">
                    <small>
                        <ul class="list-unstyled mb-0">
                            <li class="mb-2"><i class="bi bi-check-circle text-success"></i> Free pickup from your location</li>
                            <li class="mb-2"><i class="bi bi-check-circle text-success"></i> Real-time tracking</li>
                            <li class="mb-2"><i class="bi bi-check-circle text-success"></i> SMS & Email notifications</li>
                            <li class="mb-2"><i class="bi bi-check-circle text-success"></i> Insurance coverage</li>
                            <li><i class="bi bi-check-circle text-success"></i> 24/7 customer support</li>
                        </ul>
                    </small>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Real-time price calculation using district-based pricing
    function calculatePrice() {
        const weight = parseFloat(document.getElementById('weight').value) || 0;
        const deliveryType = document.getElementById('delivery_type').value;
        const vehicleType = document.getElementById('vehicle_type').value;
        const pickupDistrictId = document.getElementById('pickup_district_id').value;
        const dropDistrictId = document.getElementById('drop_district_id').value;
        
        if (weight > 0 && deliveryType && vehicleType && pickupDistrictId && dropDistrictId) {
            // Show loading state
            document.getElementById('priceEstimation').innerHTML = `
                <div class="text-center">
                    <div class="spinner-border spinner-border-sm text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 mb-0 text-muted">Calculating price...</p>
                </div>
            `;
            
            // Make API call to get district-based pricing
            fetch('/api/pricing?' + new URLSearchParams({
                pickup_district_id: pickupDistrictId,
                drop_district_id: dropDistrictId,
                service_type: deliveryType,
                weight: weight
            }), {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json',
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const pricing = data.pricing;
                    document.getElementById('priceEstimation').innerHTML = `
                        <div class="text-center">
                            <h3 class="text-success mb-3">PKR ${pricing.calculated_price}</h3>
                            <div class="small text-muted mb-3">
                                <div class="d-flex justify-content-between">
                                    <span>Base Rate:</span>
                                    <span>PKR ${pricing.base_rate}</span>
                                </div>
                                <div class="d-flex justify-content-between">
                                    <span>Per KG (${weight}kg):</span>
                                    <span>PKR ${(pricing.per_kg_rate * weight).toFixed(2)}</span>
                                </div>
                                <div class="d-flex justify-content-between">
                                    <span>Service:</span>
                                    <span>${pricing.service_type}</span>
                                </div>
                                <div class="d-flex justify-content-between">
                                    <span>Route:</span>
                                    <span>${pricing.pickup_district} → ${pricing.drop_district}</span>
                                </div>
                                <hr class="my-2">
                                <div class="d-flex justify-content-between">
                                    <span>Min Charge:</span>
                                    <span>PKR ${pricing.minimum_charge}</span>
                                </div>
                                ${pricing.maximum_charge ? `
                                <div class="d-flex justify-content-between">
                                    <span>Max Charge:</span>
                                    <span>PKR ${pricing.maximum_charge}</span>
                                </div>
                                ` : ''}
                            </div>
                            <div class="alert alert-info py-2 mb-0">
                                <small><i class="bi bi-clock"></i> Estimated delivery: ${pricing.estimated_days} day(s)</small>
                            </div>
                        </div>
                    `;
                } else {
                    // Fallback to old pricing system
                    calculateFallbackPrice(weight, deliveryType, vehicleType);
                }
            })
            .catch(error => {
                console.error('Error calculating price:', error);
                // Fallback to old pricing system
                calculateFallbackPrice(weight, deliveryType, vehicleType);
            });
        } else {
            document.getElementById('priceEstimation').innerHTML = '<p class="text-muted">Fill all required fields to get price estimation</p>';
        }
    }
    
    // Fallback pricing calculation (original logic)
    function calculateFallbackPrice(weight, deliveryType, vehicleType) {
        let basePrice = 100;
        let weightMultiplier = weight * 10;
        
        const deliveryMultiplier = {
            'standard': 1,
            'express': 1.5,
            'same_day': 2
        };
        
        const vehicleMultiplier = {
            'bike': 1,
            'car': 1.2,
            'van': 1.5,
            'truck': 2
        };
        
        let price = basePrice + weightMultiplier;
        price *= deliveryMultiplier[deliveryType] || 1;
        price *= vehicleMultiplier[vehicleType] || 1;
        
        const finalPrice = Math.round(price * 100) / 100;
        
        document.getElementById('priceEstimation').innerHTML = `
            <div class="text-center">
                <h3 class="text-warning mb-3">PKR ${finalPrice}</h3>
                <div class="small text-muted mb-3">
                    <div>Base Price: PKR ${basePrice}</div>
                    <div>Weight (${weight}kg): PKR ${weightMultiplier}</div>
                    <div>Delivery: ${deliveryType} (${deliveryMultiplier[deliveryType]}x)</div>
                    <div>Vehicle: ${vehicleType} (${vehicleMultiplier[vehicleType]}x)</div>
                </div>
                <div class="alert alert-warning py-2 mb-0">
                    <small><i class="bi bi-exclamation-triangle"></i> Using standard rates (district pricing not available)</small>
                </div>
            </div>
        `;
    }
    
    // Add event listeners for real-time calculation
    document.getElementById('weight').addEventListener('input', calculatePrice);
    document.getElementById('delivery_type').addEventListener('change', calculatePrice);
    document.getElementById('vehicle_type').addEventListener('change', calculatePrice);
    document.getElementById('pickup_district_id').addEventListener('change', calculatePrice);
    document.getElementById('drop_district_id').addEventListener('change', calculatePrice);
</script>
@endpush