@extends('layouts.mobile')

@section('title', 'Track Package')

@section('content')
<!-- Tracking Header -->
<div class="mobile-card">
    <div class="mobile-card-header text-center">
        <h2 class="mb-2">
            <i class="bi bi-search me-2"></i>
            {{ __('mobile.track_your_package') }}
        </h2>
        <p class="mb-0 opacity-75">{{ __('mobile.enter_tracking_real_time') }}</p>
    </div>
    <div class="mobile-card-body">
        <form action="{{ route('tracking.search') }}" method="POST">
            @csrf
            <div class="mb-4">
                <label for="tracking_code" class="form-label fw-bold">
                    <i class="bi bi-qr-code text-primary me-2"></i>Tracking Code
                </label>
                <div class="input-group">
                    <input type="text" class="form-control form-control-mobile @error('tracking_code') is-invalid @enderror" 
                           id="tracking_code" name="tracking_code" 
                           value="{{ old('tracking_code', request('code')) }}" 
                           placeholder="e.g., TCS20250116001" 
                           required autofocus>
                    <button class="btn btn-outline-secondary" type="button" id="scanQR">
                        <i class="bi bi-qr-code-scan"></i>
                    </button>
                </div>
                @error('tracking_code')
                    <div class="invalid-feedback">{{ $message }}</div>
                @enderror
                <small class="form-text text-muted">
                    <i class="bi bi-info-circle me-1"></i>
                    Find your tracking code on your booking confirmation
                </small>
            </div>
            
            <div class="d-grid">
                <button type="submit" class="btn btn-mobile btn-mobile-primary">
                    <i class="bi bi-search me-2"></i>
                    Track Package
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Quick Track Examples -->
<div class="mobile-card">
    <div class="mobile-card-body">
        <h6 class="mb-3">
            <i class="bi bi-lightning-charge text-warning me-2"></i>
            Quick Examples
        </h6>
        <div class="row g-2">
            <div class="col-6">
                <button class="btn btn-mobile btn-mobile-primary w-100" onclick="fillTrackingCode('TCS20250116001')">
                    <small>TCS20250116001</small>
                </button>
            </div>
            <div class="col-6">
                <button class="btn btn-mobile btn-mobile-success w-100" onclick="fillTrackingCode('TCS20250116002')">
                    <small>TCS20250116002</small>
                </button>
            </div>
        </div>
        <p class="text-muted small mt-3 mb-0">
            <i class="bi bi-info-circle me-1"></i>
            Tap on an example to try tracking
        </p>
    </div>
</div>

<!-- Tracking Features -->
<div class="mobile-card">
    <div class="mobile-card-body">
        <h6 class="mb-4">
            <i class="bi bi-star text-warning me-2"></i>
            Real-time Tracking Features
        </h6>
        <div class="row g-3">
            <div class="col-12">
                <div class="d-flex align-items-center p-3 bg-light rounded-3">
                    <div class="bg-primary text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px;">
                        <i class="bi bi-geo-alt"></i>
                    </div>
                    <div>
                        <h6 class="mb-1">Live Location</h6>
                        <small class="text-muted">See exactly where your package is right now</small>
                    </div>
                </div>
            </div>
            <div class="col-12">
                <div class="d-flex align-items-center p-3 bg-light rounded-3">
                    <div class="bg-success text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px;">
                        <i class="bi bi-clock-history"></i>
                    </div>
                    <div>
                        <h6 class="mb-1">Status Updates</h6>
                        <small class="text-muted">Get notified at every step of delivery</small>
                    </div>
                </div>
            </div>
            <div class="col-12">
                <div class="d-flex align-items-center p-3 bg-light rounded-3">
                    <div class="bg-info text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px;">
                        <i class="bi bi-person-badge"></i>
                    </div>
                    <div>
                        <h6 class="mb-1">Agent Details</h6>
                        <small class="text-muted">Contact information of your delivery agent</small>
                    </div>
                </div>
            </div>
            <div class="col-12">
                <div class="d-flex align-items-center p-3 bg-light rounded-3">
                    <div class="bg-warning text-white rounded-circle d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px;">
                        <i class="bi bi-clock"></i>
                    </div>
                    <div>
                        <h6 class="mb-1">Delivery Time</h6>
                        <small class="text-muted">Estimated delivery date and time</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@auth
<!-- Recent Orders -->
@if(auth()->user()->orders()->count() > 0)
<div class="mobile-card">
    <div class="mobile-card-body">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h6 class="mb-0">
                <i class="bi bi-clock-history text-secondary me-2"></i>
                Your Recent Orders
            </h6>
            <a href="{{ route('orders.index') }}" class="btn btn-sm btn-outline-primary">
                View All
            </a>
        </div>
        
        @foreach(auth()->user()->orders()->latest()->take(3)->get() as $order)
        <div class="border rounded-3 p-3 mb-3 bg-light" onclick="trackOrder('{{ $order->tracking_code }}')" style="cursor: pointer;">
            <div class="d-flex justify-content-between align-items-start mb-2">
                <div>
                    <strong class="text-primary">{{ $order->tracking_code }}</strong>
                    <br>
                    <small class="text-muted">
                        {{ $order->pickupDistrict->name }} → {{ $order->dropDistrict->name }}
                    </small>
                </div>
                <span class="badge bg-{{ $order->status === 'delivered' ? 'success' : ($order->status === 'in_transit' ? 'info' : 'warning') }}">
                    {{ ucfirst(str_replace('_', ' ', $order->status)) }}
                </span>
            </div>
            <div class="d-flex justify-content-between align-items-center">
                <small class="text-muted">
                    <i class="bi bi-calendar me-1"></i>
                    {{ $order->created_at->format('M d, Y') }}
                </small>
                <small class="text-primary">
                    <i class="bi bi-arrow-right-circle"></i> Tap to track
                </small>
            </div>
        </div>
        @endforeach
    </div>
</div>
@endif
@endauth

<!-- Help Section -->
<div class="mobile-card">
    <div class="mobile-card-body">
        <h6 class="mb-4">
            <i class="bi bi-question-circle text-info me-2"></i>
            Need Help?
        </h6>
        
        <div class="accordion" id="helpAccordion">
            <div class="accordion-item border-0 mb-2">
                <h2 class="accordion-header">
                    <button class="accordion-button collapsed bg-light" type="button" data-bs-toggle="collapse" data-bs-target="#help1">
                        <i class="bi bi-search me-2"></i>
                        Where do I find my tracking code?
                    </button>
                </h2>
                <div id="help1" class="accordion-collapse collapse" data-bs-parent="#helpAccordion">
                    <div class="accordion-body">
                        Your tracking code is provided when you book a courier service. You can find it in:
                        <ul class="mt-2">
                            <li>Booking confirmation email</li>
                            <li>SMS notification</li>
                            <li>Your orders page</li>
                            <li>Booking receipt</li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <div class="accordion-item border-0 mb-2">
                <h2 class="accordion-header">
                    <button class="accordion-button collapsed bg-light" type="button" data-bs-toggle="collapse" data-bs-target="#help2">
                        <i class="bi bi-clock me-2"></i>
                        How often is tracking updated?
                    </button>
                </h2>
                <div id="help2" class="accordion-collapse collapse" data-bs-parent="#helpAccordion">
                    <div class="accordion-body">
                        Tracking information is updated in real-time as your package moves through our network. Updates typically occur:
                        <ul class="mt-2">
                            <li>When package is picked up</li>
                            <li>At sorting facilities</li>
                            <li>During transit</li>
                            <li>Upon delivery</li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <div class="accordion-item border-0">
                <h2 class="accordion-header">
                    <button class="accordion-button collapsed bg-light" type="button" data-bs-toggle="collapse" data-bs-target="#help3">
                        <i class="bi bi-telephone me-2"></i>
                        Package not updating?
                    </button>
                </h2>
                <div id="help3" class="accordion-collapse collapse" data-bs-parent="#helpAccordion">
                    <div class="accordion-body">
                        If your tracking hasn't updated for more than 24 hours, please contact us:
                        <div class="row g-2 mt-2">
                            <div class="col-6">
                                <a href="tel:+923001234567" class="btn btn-outline-success w-100 btn-sm">
                                    <i class="bi bi-telephone"></i> Call
                                </a>
                            </div>
                            <div class="col-6">
                                <a href="mailto:support@tcscourier.com" class="btn btn-outline-info w-100 btn-sm">
                                    <i class="bi bi-envelope"></i> Email
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Fill tracking code from examples
    function fillTrackingCode(code) {
        document.getElementById('tracking_code').value = code;
        if ('vibrate' in navigator) {
            navigator.vibrate(50);
        }
    }
    
    // Track order from recent orders
    function trackOrder(code) {
        document.getElementById('tracking_code').value = code;
        document.querySelector('form').submit();
    }
    
    // QR Code scanner (placeholder - would need camera API)
    document.getElementById('scanQR').addEventListener('click', function() {
        if ('vibrate' in navigator) {
            navigator.vibrate(100);
        }
        
        // Placeholder for QR scanner
        alert('QR Scanner feature coming soon! For now, please enter your tracking code manually.');
        
        // In a real implementation, you would use:
        // - getUserMedia API for camera access
        // - QR code scanning library like jsQR
        // - Handle camera permissions
    });
    
    // Auto-format tracking code
    document.getElementById('tracking_code').addEventListener('input', function(e) {
        let value = e.target.value.toUpperCase();
        // Remove any non-alphanumeric characters
        value = value.replace(/[^A-Z0-9]/g, '');
        e.target.value = value;
    });
    
    // Form validation and loading state
    document.querySelector('form').addEventListener('submit', function(e) {
        const trackingCode = document.getElementById('tracking_code').value;
        
        if (!trackingCode || trackingCode.length < 5) {
            e.preventDefault();
            
            const alertDiv = document.createElement('div');
            alertDiv.className = 'alert alert-warning alert-mobile mt-3';
            alertDiv.innerHTML = '<i class="bi bi-exclamation-triangle me-2"></i>Please enter a valid tracking code.';
            
            const form = document.querySelector('form');
            form.appendChild(alertDiv);
            
            setTimeout(() => {
                alertDiv.remove();
            }, 3000);
            
            return;
        }
        
        // Show loading state
        const submitBtn = document.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="bi bi-hourglass-split me-2"></i>Tracking...';
        submitBtn.disabled = true;
        
        showLoading();
    });
    
    // Auto-focus on tracking input for desktop
    document.addEventListener('DOMContentLoaded', function() {
        if (!window.matchMedia('(max-width: 768px)').matches) {
            setTimeout(() => {
                document.getElementById('tracking_code').focus();
            }, 500);
        }
    });
    
    // Add haptic feedback to interactive elements
    document.querySelectorAll('.btn, .accordion-button').forEach(element => {
        element.addEventListener('click', function() {
            if ('vibrate' in navigator) {
                navigator.vibrate(50);
            }
        });
    });
</script>
@endpush