<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\CourierController;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Controllers\Admin\AdminController;
use App\Http\Controllers\Admin\AdminAuthController;
use App\Http\Controllers\MobileController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
*/

// Home Routes
Route::get('/', function(\Illuminate\Http\Request $request) {
    // Check if mobile device
    $mobileController = new MobileController();
    if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
        return $mobileController->home($request);
    }
    return app(HomeController::class)->index();
})->name('home');

Route::get('/tracking', function(\Illuminate\Http\Request $request) {
    $mobileController = new MobileController();
    if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
        return $mobileController->tracking();
    }
    return app(HomeController::class)->tracking();
})->name('tracking');

Route::post('/tracking/search', function(\Illuminate\Http\Request $request) {
    $mobileController = new MobileController();
    if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
        return $mobileController->trackingSearch($request);
    }
    return app(HomeController::class)->trackingSearch($request);
})->name('tracking.search');

// Mobile-specific routes
Route::prefix('mobile')->name('mobile.')->group(function () {
    Route::get('/', [MobileController::class, 'home'])->name('home');
    Route::get('/login', [MobileController::class, 'showLoginForm'])->name('login');
    Route::get('/tracking', [MobileController::class, 'tracking'])->name('tracking');
    Route::post('/tracking/search', [MobileController::class, 'trackingSearch'])->name('tracking.search');
    Route::get('/manifest.json', [MobileController::class, 'manifest'])->name('manifest');
    
    Route::middleware('auth')->group(function () {
        Route::get('/profile', [MobileController::class, 'profile'])->name('profile');
        Route::put('/profile', [MobileController::class, 'updateProfile'])->name('profile.update');
        Route::get('/orders', [MobileController::class, 'orders'])->name('orders');
        Route::get('/orders/{order}', [MobileController::class, 'showOrder'])->name('orders.show');
        Route::get('/courier/create', [MobileController::class, 'createCourier'])->name('courier.create');
        Route::post('/courier', [MobileController::class, 'storeCourier'])->name('courier.store');
    });
});

// Authentication Routes
Route::middleware('guest')->group(function () {
    Route::get('/login', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->showLoginForm();
        }
        return app(LoginController::class)->showLoginForm();
    })->name('login');
    Route::post('/login', [LoginController::class, 'login']);
    Route::get('/register', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return view('mobile.auth.register');
        }
        return app(RegisterController::class)->showRegistrationForm();
    })->name('register');
    Route::post('/register', [RegisterController::class, 'register']);
    
    // Language switching
    Route::post('/language/switch', function(\Illuminate\Http\Request $request) {
        $language = $request->input('language', 'en');
        
        // Validate language
        if (!in_array($language, ['en', 'ur'])) {
            $language = 'en';
        }
        
        // Store language preference in session
        session(['locale' => $language]);
        
        // Set app locale
        app()->setLocale($language);
        
        return redirect()->back();
    })->name('language.switch');
    
    // Social Authentication
    Route::get('/auth/google', [LoginController::class, 'redirectToGoogle'])->name('auth.google');
    Route::get('/auth/google/callback', [LoginController::class, 'handleGoogleCallback']);
    Route::get('/auth/facebook', [LoginController::class, 'redirectToFacebook'])->name('auth.facebook');
    Route::get('/auth/facebook/callback', [LoginController::class, 'handleFacebookCallback']);
    
    // Phone OTP Authentication
    Route::get('/auth/phone', [LoginController::class, 'showPhoneForm'])->name('auth.phone');
    Route::post('/auth/phone/send-otp', [LoginController::class, 'sendOTP'])->name('auth.phone.send-otp');
    Route::post('/auth/phone/verify-otp', [LoginController::class, 'verifyOTP'])->name('auth.phone.verify-otp');
    
    // Password Reset Routes
    Route::get('/forgot-password', [LoginController::class, 'showForgotPasswordForm'])->name('password.request');
    Route::post('/forgot-password', [LoginController::class, 'sendResetLink'])->name('password.email');
});

// Authenticated Routes
Route::middleware('auth')->group(function () {
    Route::post('/logout', [LoginController::class, 'logout'])->name('logout');
    
    // Profile Routes
    Route::get('/profile', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->profile();
        }
        return app(HomeController::class)->profile();
    })->name('profile');
    Route::put('/profile', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->updateProfile($request);
        }
        return app(HomeController::class)->updateProfile($request);
    })->name('profile.update');
    
    // Courier Booking Routes
    Route::get('/courier/create', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->createCourier();
        }
        return app(CourierController::class)->create();
    })->name('courier.create');
    Route::post('/courier/store', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->storeCourier($request);
        }
        return app(CourierController::class)->store($request);
    })->name('courier.store');
    Route::post('/courier', [CourierController::class, 'store'])->name('courier.store.alt');
    Route::get('/courier/{order}/confirmation', [CourierController::class, 'confirmation'])->name('courier.confirmation');
    
    // Order Management Routes
    Route::get('/orders', function(\Illuminate\Http\Request $request) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->orders($request);
        }
        return app(CourierController::class)->index();
    })->name('orders.index');
    Route::get('/orders/{order}', function(\Illuminate\Http\Request $request, $order) {
        $mobileController = new MobileController();
        if ($mobileController->isMobile($request) || $request->get('mobile') === '1') {
            return $mobileController->showOrder($order);
        }
        return app(CourierController::class)->show($order);
    })->name('orders.show');
    Route::get('/orders/{order}/invoice', [CourierController::class, 'invoice'])->name('orders.invoice');
    
    // Payment Routes
    Route::post('/payment/jazzcash/{order}', [CourierController::class, 'processJazzCash'])->name('payment.jazzcash');
    Route::post('/payment/easypaisa/{order}', [CourierController::class, 'processEasypaisa'])->name('payment.easypaisa');
    Route::get('/payment/success/{order}', [CourierController::class, 'paymentSuccess'])->name('payment.success');
    Route::get('/payment/failed/{order}', [CourierController::class, 'paymentFailed'])->name('payment.failed');
});

// Admin Authentication Routes
Route::prefix('admin')->name('admin.')->group(function () {
    Route::middleware('guest')->group(function () {
        Route::get('/login', [AdminAuthController::class, 'showLoginForm'])->name('login');
        Route::post('/login', [AdminAuthController::class, 'login'])->name('login.submit');
        Route::get('/forgot-password', [AdminAuthController::class, 'showForgotPasswordForm'])->name('password.request');
        Route::post('/forgot-password', [AdminAuthController::class, 'sendResetLink'])->name('password.email');
    });
    
    Route::middleware('auth')->group(function () {
        Route::post('/logout', [AdminAuthController::class, 'logout'])->name('logout');
        Route::post('/verify-password', [AdminAuthController::class, 'verifyPassword'])->name('verify-password');
        Route::get('/session-info', [AdminAuthController::class, 'getSessionInfo'])->name('session-info');
    });
});

// Admin Routes
Route::middleware(['auth', 'admin'])->prefix('admin')->name('admin.')->group(function () {
    Route::get('/dashboard', [AdminController::class, 'dashboard'])->name('dashboard');
    
    // Order Management
    Route::get('/orders', [AdminController::class, 'orders'])->name('orders');
    Route::get('/orders/{order}', [AdminController::class, 'showOrder'])->name('orders.show');
    Route::put('/orders/{order}/status', [AdminController::class, 'updateOrderStatus'])->name('orders.update-status');
    Route::put('/orders/{order}/assign-agent', [AdminController::class, 'assignAgent'])->name('orders.assign-agent');
    
    // Agent Management
    Route::get('/agents', [AdminController::class, 'agents'])->name('agents');
    Route::get('/agents/create', [AdminController::class, 'createAgent'])->name('agents.create');
    Route::post('/agents', [AdminController::class, 'storeAgent'])->name('agents.store');
    Route::get('/agents/{agent}/edit', [AdminController::class, 'editAgent'])->name('agents.edit');
    Route::put('/agents/{agent}', [AdminController::class, 'updateAgent'])->name('agents.update');
    
    // Office Management
    Route::get('/offices', [AdminController::class, 'offices'])->name('offices');
    Route::get('/offices/create', [AdminController::class, 'createOffice'])->name('offices.create');
    Route::post('/offices', [AdminController::class, 'storeOffice'])->name('offices.store');
    Route::get('/offices/{office}/edit', [AdminController::class, 'editOffice'])->name('offices.edit');
    Route::put('/offices/{office}', [AdminController::class, 'updateOffice'])->name('offices.update');
    
    // District Management
    Route::get('/districts', [AdminController::class, 'districts'])->name('districts');
    Route::get('/districts/create', [AdminController::class, 'createDistrict'])->name('districts.create');
    Route::post('/districts', [AdminController::class, 'storeDistrict'])->name('districts.store');
    Route::get('/districts/{district}/edit', [AdminController::class, 'editDistrict'])->name('districts.edit');
    Route::put('/districts/{district}', [AdminController::class, 'updateDistrict'])->name('districts.update');
    Route::delete('/districts/{district}', [AdminController::class, 'destroyDistrict'])->name('districts.destroy');
    
    // User Management
    Route::get('/users', [AdminController::class, 'users'])->name('users');
    Route::get('/users/{user}/edit', [AdminController::class, 'editUser'])->name('users.edit');
    Route::put('/users/{user}', [AdminController::class, 'updateUser'])->name('users.update');
    Route::patch('/users/{user}/toggle-status', [AdminController::class, 'toggleUserStatus'])->name('users.toggle-status');
    Route::delete('/users/{user}', [AdminController::class, 'destroyUser'])->name('users.destroy');
    
    // District Pricing Management
    Route::get('/pricing', [AdminController::class, 'districtPricing'])->name('pricing');
    Route::get('/pricing/create', [AdminController::class, 'createDistrictPricing'])->name('pricing.create');
    Route::post('/pricing', [AdminController::class, 'storeDistrictPricing'])->name('pricing.store');
    Route::get('/pricing/{pricing}/edit', [AdminController::class, 'editDistrictPricing'])->name('pricing.edit');
    Route::put('/pricing/{pricing}', [AdminController::class, 'updateDistrictPricing'])->name('pricing.update');
    Route::delete('/pricing/{pricing}', [AdminController::class, 'destroyDistrictPricing'])->name('pricing.destroy');
    Route::patch('/pricing/{pricing}/toggle-status', [AdminController::class, 'togglePricingStatus'])->name('pricing.toggle-status');
    
    // Reports
    Route::get('/reports', [AdminController::class, 'reports'])->name('reports');
    Route::get('/reports/revenue', [AdminController::class, 'revenueReport'])->name('reports.revenue');
    Route::get('/reports/performance', [AdminController::class, 'performanceReport'])->name('reports.performance');
});

// API Routes for AJAX requests
Route::middleware('auth')->prefix('api')->group(function () {
    Route::get('/districts', [CourierController::class, 'getDistricts']);
    Route::get('/offices/{district}', [CourierController::class, 'getOfficesByDistrict']);
    Route::post('/calculate-price', [CourierController::class, 'calculatePrice']);
    Route::get('/pricing', [AdminController::class, 'getPricing']); // For getting district pricing
});
